<?php
/**
 * Studiare Quiz Admin Results
 * Handles the admin page for displaying quiz results using WP_List_Table
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Ensure WP_List_Table is loaded
if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Custom WP_List_Table class for quiz results
 */
class Studiare_Quiz_Results_Table extends WP_List_Table {
    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct([
            'singular' => 'quiz_result',
            'plural'   => 'quiz_results',
            'ajax'     => false
        ]);
    }

    /**
     * Define table columns
     *
     * @return array
     */
    public function get_columns() {
        return [
            'cb'            => '<input type="checkbox" />',
            'quiz_title'    => __('Quiz', 'studiare-core'),
            'product_title' => __('Product/Course', 'studiare-core'),
            'user_name'     => __('User', 'studiare-core'),
            'score'         => __('Score', 'studiare-core'),
            'timestamp'     => __('Date', 'studiare-core'),
            'status'        => __('Status', 'studiare-core'),
            'details'       => __('Details', 'studiare-core')
        ];
    }

    /**
     * Define sortable columns
     *
     * @return array
     */
    public function get_sortable_columns() {
        return [
            'quiz_title'    => ['quiz_title', false],
            'product_title' => ['product_title', false],
            'user_name'     => ['user_name', false],
            'score'         => ['score', true],
            'timestamp'     => ['timestamp', true]
        ];
    }

    /**
     * Define bulk actions
     *
     * @return array
     */
    public function get_bulk_actions() {
        return [
            'delete' => __('Delete', 'studiare-core')
        ];
    }

    /**
     * Handle bulk actions
     */
    public function process_bulk_action() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'studiare_quiz_results';

        if ('delete' === $this->current_action()) {
            $nonce = isset($_REQUEST['_wpnonce']) ? sanitize_text_field($_REQUEST['_wpnonce']) : '';
            if (!wp_verify_nonce($nonce, 'bulk-' . $this->_args['plural'])) {
                wp_die(__('Security check failed.', 'studiare-core'));
            }

            $ids = isset($_REQUEST['quiz_result']) ? array_map('intval', (array)$_REQUEST['quiz_result']) : [];
            if (!empty($ids)) {
                $ids_placeholder = implode(',', array_fill(0, count($ids), '%d'));
                $wpdb->query($wpdb->prepare("DELETE FROM $table_name WHERE id IN ($ids_placeholder)", ...$ids));
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Selected quiz results have been deleted.', 'studiare-core') . '</p></div>';
            }
        }
    }

    /**
     * Prepare table items
     */
    public function prepare_items() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'studiare_quiz_results';
        $per_page = 20;
        $current_page = $this->get_pagenum();

        // Handle sorting
        $orderby = !empty($_GET['orderby']) ? sanitize_text_field($_GET['orderby']) : 'timestamp';
        $order = !empty($_GET['order']) ? sanitize_text_field($_GET['order']) : 'DESC';

        $column_map = [
            'quiz_title'    => 'quiz_title',
            'product_title' => 'product_title',
            'user_name'     => 'user_name',
            'score'         => 'r.score',
            'timestamp'     => 'r.timestamp'
        ];

        $db_orderby = isset($column_map[$orderby]) ? $column_map[$orderby] : 'r.timestamp';
        $db_order = in_array(strtoupper($order), ['ASC', 'DESC']) ? $order : 'DESC';

        // Handle filters and search
        $where = [];
        $params = [];

        if (!empty($_GET['filter_quiz'])) {
            $quiz_ids = $wpdb->get_col(
                $wpdb->prepare(
                    "SELECT ID FROM {$wpdb->posts} WHERE post_type = 'sc-quiz' AND post_title LIKE %s",
                    '%' . $wpdb->esc_like(sanitize_text_field($_GET['filter_quiz'])) . '%'
                )
            );
            if (!empty($quiz_ids)) {
                $where[] = 'r.quiz_id IN (' . implode(',', array_map('intval', $quiz_ids)) . ')';
            } else {
                $where[] = '1=0';
            }
        }

        if (!empty($_GET['filter_product'])) {
            $product_ids = $wpdb->get_col(
                $wpdb->prepare(
                    "SELECT ID FROM {$wpdb->posts} WHERE post_type = 'product' AND post_title LIKE %s",
                    '%' . $wpdb->esc_like(sanitize_text_field($_GET['filter_product'])) . '%'
                )
            );
            if (!empty($product_ids)) {
                $where[] = 'r.product_id IN (' . implode(',', array_map('intval', $product_ids)) . ')';
            } else {
                $where[] = '1=0';
            }
        }

        if (!empty($_GET['filter_user'])) {
            $user_ids = $wpdb->get_col(
                $wpdb->prepare(
                    "SELECT ID FROM {$wpdb->users} WHERE display_name LIKE %s",
                    '%' . $wpdb->esc_like(sanitize_text_field($_GET['filter_user'])) . '%'
                )
            );
            if (!empty($user_ids)) {
                $where[] = 'r.user_id IN (' . implode(',', array_map('intval', $user_ids)) . ')';
            } else {
                $where[] = '1=0';
            }
        }

        if (!empty($_GET['filter_status']) && in_array($_GET['filter_status'], ['passed', 'failed'])) {
            $where[] = '1=1'; // Filter status in PHP
        }

        if (!empty($_GET['s'])) {
            $search_term = '%' . $wpdb->esc_like(sanitize_text_field($_GET['s'])) . '%';
            $where[] = '(p.post_title LIKE %s OR prod.post_title LIKE %s OR u.display_name LIKE %s OR r.score LIKE %s)';
            $params[] = $search_term;
            $params[] = $search_term;
            $params[] = $search_term;
            $params[] = $search_term;
        }

        $where_clause = '';
        if (!empty($where)) {
            $where_clause = 'WHERE ' . implode(' AND ', $where);
        }

        $base_query = "SELECT COUNT(*) FROM $table_name r 
                       LEFT JOIN {$wpdb->posts} p ON r.quiz_id = p.ID AND p.post_type = 'sc-quiz'
                       LEFT JOIN {$wpdb->posts} prod ON r.product_id = prod.ID AND prod.post_type = 'product'
                       LEFT JOIN {$wpdb->users} u ON r.user_id = u.ID";

        if (!empty($where_clause)) {
            $total_items_query = "$base_query $where_clause";
        } else {
            $total_items_query = $base_query;
        }

        if (!empty($params)) {
            $total_items = $wpdb->get_var($wpdb->prepare($total_items_query, ...$params));
        } else {
            $total_items = $wpdb->get_var($total_items_query);
        }

        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => ceil($total_items / $per_page)
        ]);

        $base_select_query = "SELECT r.*, 
                             COALESCE(p.post_title, %s) AS quiz_title,
                             COALESCE(prod.post_title, %s) AS product_title,
                             COALESCE(u.display_name, %s) AS user_name 
                      FROM $table_name r 
                      LEFT JOIN {$wpdb->posts} p ON r.quiz_id = p.ID AND p.post_type = 'sc-quiz'
                      LEFT JOIN {$wpdb->posts} prod ON r.product_id = prod.ID AND prod.post_type = 'product'
                      LEFT JOIN {$wpdb->users} u ON r.user_id = u.ID";

        $query_params = [
            __('Deleted Quiz', 'studiare-core'),
            __('N/A', 'studiare-core'),
            __('Unknown User', 'studiare-core'),
            $per_page,
            ($current_page - 1) * $per_page
        ];

        if (!empty($where_clause)) {
            $query = "$base_select_query $where_clause ORDER BY $db_orderby $db_order LIMIT %d OFFSET %d";
            $query_params = array_merge($query_params, $params);
        } else {
            $query = "$base_select_query ORDER BY $db_orderby $db_order LIMIT %d OFFSET %d";
        }

        $results = $wpdb->get_results($wpdb->prepare($query, ...$query_params));

        if (!empty($_GET['filter_status']) && in_array($_GET['filter_status'], ['passed', 'failed'])) {
            $filtered_results = [];
            $desired_status = sanitize_text_field($_GET['filter_status']);
            foreach ($results as $result) {
                $details = maybe_unserialize($result->details);
                $status = isset($details['status']) ? $details['status'] : ($result->score >= 50 ? 'passed' : 'failed');
                if ($status === $desired_status) {
                    $filtered_results[] = $result;
                }
            }
            $this->items = $filtered_results;
            $this->set_pagination_args([
                'total_items' => count($filtered_results),
                'per_page'    => $per_page,
                'total_pages' => ceil(count($filtered_results) / $per_page)
            ]);
        } else {
            $this->items = $results;
        }
    }

    /**
     * Render checkbox column
     *
     * @param object $item
     * @return string
     */
    public function column_cb($item) {
        return sprintf(
            '<input type="checkbox" name="quiz_result[]" value="%s" />',
            esc_attr($item->id)
        );
    }

    /**
     * Render column data
     *
     * @param object $item
     * @param string $column_name
     * @return string
     */
    public function column_default($item, $column_name) {
        $columns = array_keys($this->get_columns());
        if (!in_array($column_name, $columns)) {
            return ''; // Prevent rendering N/A for unknown columns
        }

        switch ($column_name) {
            case 'quiz_title':
                return !empty($item->quiz_title) ? esc_html($item->quiz_title) : __('N/A', 'studiare-core');
            case 'product_title':
                return !empty($item->product_title) ? esc_html($item->product_title) : __('N/A', 'studiare-core');
            case 'user_name':
                return !empty($item->user_name) ? esc_html($item->user_name) : __('N/A', 'studiare-core');
            case 'score':
                return is_numeric($item->score) ? esc_html(number_format($item->score, 1)) : __('N/A', 'studiare-core');
            case 'timestamp':
                return !empty($item->timestamp) ? esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($item->timestamp))) : __('N/A', 'studiare-core');
            case 'status':
                $details = maybe_unserialize($item->details);
                $status = isset($details['status']) ? $details['status'] : ($item->score >= 50 ? 'passed' : 'failed');
                return sprintf(
                    '<span class="%s">%s</span>',
                    esc_attr($status === 'passed' ? 'status-passed' : 'status-failed'),
                    esc_html($status === 'passed' ? __('Passed', 'studiare-core') : __('Failed', 'studiare-core'))
                );
            case 'details':
                $details = maybe_unserialize($item->details);
                $output = '<ul>';
                $output .= '<li>' . esc_html__('Correct', 'studiare-core') . ': ' . esc_html($details['correct_percentage'] ?? 0) . '%</li>';
                $output .= '<li>' . esc_html__('Incorrect', 'studiare-core') . ': ' . esc_html($details['incorrect_percentage'] ?? 0) . '%</li>';
                $output .= '<li>' . esc_html__('Unanswered', 'studiare-core') . ': ' . esc_html($details['unanswered_percentage'] ?? 0) . '%</li>';
                $output .= '</ul>';
                return $output;
            default:
                return ''; // Do not display N/A for unknown columns
        }
    }

    /**
     * Message to display when no items are found
     */
    public function no_items() {
        ?>
        <tr class="no-items">
            <td colspan="<?php echo count($this->get_columns()); ?>">
                <?php esc_html_e('No quiz results found.', 'studiare-core'); ?>
            </td>
        </tr>
        <?php
    }

    /**
     * Override single_row_columns to debug rendering
     */
    public function single_row_columns($item) {
        $columns = $this->get_columns();
        foreach ($columns as $column_name => $column_display_name) {
            $class = "class='$column_name column-$column_name'";
            $attributes = "$class";
            echo "<td $attributes>";
            if ($column_name === 'cb') {
                echo $this->column_cb($item);
            } else {
                echo $this->column_default($item, $column_name);
            }
            echo "</td>";
        }
    }

    /**
     * Override display_rows to ensure proper rendering
     */
    public function display_rows() {
        foreach ($this->items as $item) {
            if (!isset($item->id)) {
                continue;
            }
            echo '<tr>';
            $this->single_row_columns($item);
            echo '</tr>';
        }
    }

    /**
     * Override print_column_headers to ensure headers are rendered
     */
    public function print_column_headers($with_id = true) {
        $columns = $this->get_columns();
        if (empty($columns)) {
            return;
        }

        $sortable_columns = $this->get_sortable_columns();
        $current_url = esc_url(set_url_scheme('http://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']));
        $current_orderby = isset($_GET['orderby']) ? sanitize_text_field($_GET['orderby']) : '';
        $current_order = isset($_GET['order']) ? sanitize_text_field($_GET['order']) : 'asc';

        foreach ($columns as $column_key => $column_display_name) {
            $class = "manage-column column-$column_key";
            $style = '';

            if (in_array($column_key, array())) {
                $style .= 'display:none;';
            }

            if (isset($sortable_columns[$column_key])) {
                $class .= ' sortable';
                $order = ($current_orderby === $column_key && $current_order === 'asc') ? 'desc' : 'asc';
                $url = add_query_arg(['orderby' => $column_key, 'order' => $order], $current_url);
                $column_display_name = sprintf(
                    '<a href="%s"><span>%s</span><span class="sorting-indicator"></span></a>',
                    esc_url($url),
                    $column_display_name
                );
                $class .= ($current_orderby === $column_key) ? ' sorted ' . $current_order : '';
            }

            if ($column_key === 'cb') {
                $class .= ' check-column';
                $column_display_name = '<input type="checkbox" />';
            }

            $id = $with_id ? "id='$column_key'" : '';
            echo "<th scope='col' $id class='$class' style='$style'>$column_display_name</th>";
        }
    }

    /**
     * Display rows or placeholder if no items
     */
    public function display_rows_or_placeholder() {
        if (empty($this->items)) {
            $this->no_items();
            return;
        }

        $this->display_rows();
    }

    /**
     * Override display to ensure proper table structure
     */
    public function display() {
        $singular = $this->_args['singular'];
        $this->display_tablenav('top');
        ?>
        <table class="wp-list-table <?php echo implode(' ', $this->get_table_classes()); ?>">
            <thead>
                <tr>
                    <?php $this->print_column_headers(); ?>
                </tr>
            </thead>
            <tbody id="the-list"<?php if ($singular) echo " data-wp-lists='list:$singular'"; ?>>
                <?php $this->display_rows_or_placeholder(); ?>
            </tbody>
            <tfoot>
                <tr>
                    <?php $this->print_column_headers(false); ?>
                </tr>
            </tfoot>
        </table>
        <?php
        $this->display_tablenav('bottom');
    }
}

/**
 * Add admin menu for quiz results
 */
function studiare_quiz_add_admin_menu() {
    add_menu_page(
        __('Quiz Results', 'studiare-core'),
        __('Quiz Results', 'studiare-core'),
        'manage_options',
        'studiare-quiz-results',
        'studiare_quiz_render_results_page',
        'dashicons-chart-bar',
        30
    );
}
add_action('admin_menu', 'studiare_quiz_add_admin_menu');

/**
 * Add custom styles to ensure headers are visible and align with WordPress style
 */
function studiare_quiz_admin_styles() {
    echo '<style>
        .status-passed {
            color: #4CAF50;
        }
        .status-failed {
            color: #F44336;
        }
        .add-new-form {
            margin-bottom: 20px;
            padding: 15px;
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        .add-new-form label {
            display: inline-block;
            width: 120px;
            font-weight: bold;
        }
        .add-new-form select,
        .add-new-form input[type="number"],
        .add-new-form input[type="datetime-local"] {
            width: 200px;
            margin-bottom: 10px;
        }
        .add-new-form input[type="submit"] {
            margin-top: 10px;
        }
    </style>';
}
add_action('admin_head', 'studiare_quiz_admin_styles');

/**
 * Enqueue admin scripts
 */
function studiare_quiz_enqueue_admin_scripts($hook) {
    if ('toplevel_page_studiare-quiz-results' !== $hook) {
        return;
    }

    $js_file = STUDIARE_QUIZ_PATH . '/assets/js/studiare-quiz-admin.js';
    if (file_exists($js_file)) {
        wp_enqueue_script(
            'studiare-quiz-admin',
            STUDIARE_QUIZ_URL . '/assets/js/studiare-quiz-admin.js',
            ['jquery'],
            '1.0.0',
            true
        );
    }
}
add_action('admin_enqueue_scripts', 'studiare_quiz_enqueue_admin_scripts');

/**
 * Handle adding new quiz result
 */
function studiare_quiz_handle_add_new() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'studiare_quiz_results';

    if (isset($_POST['add_new_quiz_result']) && check_admin_referer('add_new_quiz_result_nonce')) {
        $quiz_id = isset($_POST['quiz_id']) ? intval($_POST['quiz_id']) : 0;
        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        $score = isset($_POST['score']) ? floatval($_POST['score']) : 0;
        $correct_percentage = isset($_POST['correct_percentage']) ? floatval($_POST['correct_percentage']) : $score;
        $incorrect_percentage = isset($_POST['incorrect_percentage']) ? floatval($_POST['incorrect_percentage']) : (100 - $score);
        $unanswered_percentage = isset($_POST['unanswered_percentage']) ? floatval($_POST['unanswered_percentage']) : 0;
        $timestamp = isset($_POST['timestamp']) ? sanitize_text_field($_POST['timestamp']) : current_time('mysql');
        $status = $score >= 50 ? 'passed' : 'failed';

        // Validate percentages
        $total_percentage = $correct_percentage + $incorrect_percentage + $unanswered_percentage;
        if (abs($total_percentage - 100) > 0.1) { // Allow small rounding errors
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Error: The sum of correct, incorrect, and unanswered percentages must equal 100%.', 'studiare-core') . '</p></div>';
            return;
        }

        $details = serialize([
            'total_possible_score' => 100,
            'correct_percentage' => $correct_percentage,
            'incorrect_percentage' => $incorrect_percentage,
            'unanswered_percentage' => $unanswered_percentage,
            'is_highest_score' => false,
            'status' => $status,
            'passing_score' => 50
        ]);

        $inserted = $wpdb->insert(
            $table_name,
            [
                'quiz_id' => $quiz_id,
                'product_id' => $product_id,
                'user_id' => $user_id,
                'score' => $score,
                'timestamp' => $timestamp,
                'details' => $details
            ],
            ['%d', '%d', '%d', '%f', '%s', '%s']
        );

        if ($inserted) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Quiz result added successfully.', 'studiare-core') . '</p></div>';
        } else {
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Failed to add quiz result.', 'studiare-core') . '</p></div>';
        }
    }
}

/**
 * Render the quiz results page
 */
function studiare_quiz_render_results_page() {
    studiare_quiz_handle_add_new();

    $table = new Studiare_Quiz_Results_Table();
    $table->process_bulk_action();
    $table->prepare_items();

    global $wpdb;
    $quizzes = $wpdb->get_results("SELECT ID, post_title FROM {$wpdb->posts} WHERE post_type = 'sc-quiz' AND post_status = 'publish'");
    $products = $wpdb->get_results("SELECT ID, post_title FROM {$wpdb->posts} WHERE post_type = 'product' AND post_status = 'publish'");
    $users = $wpdb->get_results("SELECT ID, display_name FROM {$wpdb->users}");
    ?>
    <div class="wrap">
        <h1 class="wp-heading-inline"><?php esc_html_e('Quiz Results', 'studiare-core'); ?></h1>
        <a href="#" class="page-title-action" onclick="document.getElementById('add-new-form').style.display='block'; return false;"><?php esc_html_e('Add New', 'studiare-core'); ?></a>
        <hr class="wp-header-end">

        <div id="add-new-form" class="add-new-form" style="display: none;">
            <form method="post">
                <?php wp_nonce_field('add_new_quiz_result_nonce'); ?>
                <input type="hidden" name="add_new_quiz_result" value="1">
                
                <label for="quiz_id"><?php esc_html_e('Quiz', 'studiare-core'); ?></label>
                <select name="quiz_id" id="quiz_id" required>
                    <option value=""><?php esc_html_e('Select a quiz', 'studiare-core'); ?></option>
                    <?php foreach ($quizzes as $quiz) : ?>
                        <option value="<?php echo esc_attr($quiz->ID); ?>"><?php echo esc_html($quiz->post_title); ?></option>
                    <?php endforeach; ?>
                </select><br>

                <label for="product_id"><?php esc_html_e('Product/Course', 'studiare-core'); ?></label>
                <select name="product_id" id="product_id">
                    <option value="0"><?php esc_html_e('No Product', 'studiare-core'); ?></option>
                    <?php foreach ($products as $product) : ?>
                        <option value="<?php echo esc_attr($product->ID); ?>"><?php echo esc_html($product->post_title); ?></option>
                    <?php endforeach; ?>
                </select><br>

                <label for="user_id"><?php esc_html_e('User', 'studiare-core'); ?></label>
                <select name="user_id" id="user_id" required>
                    <option value=""><?php esc_html_e('Select a user', 'studiare-core'); ?></option>
                    <?php foreach ($users as $user) : ?>
                        <option value="<?php echo esc_attr($user->ID); ?>"><?php echo esc_html($user->display_name); ?></option>
                    <?php endforeach; ?>
                </select><br>

                <label for="score"><?php esc_html_e('Score', 'studiare-core'); ?></label>
                <input type="number" name="score" id="score" min="0" max="100" step="0.1" required><br>

                <label for="correct_percentage"><?php esc_html_e('Correct Percentage', 'studiare-core'); ?></label>
                <input type="number" name="correct_percentage" id="correct_percentage" min="0" max="100" step="0.1" placeholder="Default: Same as score"><br>

                <label for="incorrect_percentage"><?php esc_html_e('Incorrect Percentage', 'studiare-core'); ?></label>
                <input type="number" name="incorrect_percentage" id="incorrect_percentage" min="0" max="100" step="0.1" placeholder="Default: 100 - score"><br>

                <label for="unanswered_percentage"><?php esc_html_e('Unanswered Percentage', 'studiare-core'); ?></label>
                <input type="number" name="unanswered_percentage" id="unanswered_percentage" min="0" max="100" step="0.1" value="0"><br>

                <label for="timestamp"><?php esc_html_e('Date', 'studiare-core'); ?></label>
                <input type="datetime-local" name="timestamp" id="timestamp" value="<?php echo esc_attr(current_time('Y-m-d\TH:i')); ?>"><br>

                <input type="submit" class="button button-primary" value="<?php esc_attr_e('Add Quiz Result', 'studiare-core'); ?>">
            </form>
        </div>

        <form method="get">
            <input type="hidden" name="page" value="studiare-quiz-results">
            <?php $table->search_box(__('Search', 'studiare-core'), 'search_id'); ?>
            <div class="tablenav top">
                <div class="alignleft actions">
                    <label for="filter_quiz"><?php esc_html_e('Filter by Quiz', 'studiare-core'); ?></label>
                    <input type="text" name="filter_quiz" id="filter_quiz" value="<?php echo isset($_GET['filter_quiz']) ? esc_attr(sanitize_text_field($_GET['filter_quiz'])) : ''; ?>" placeholder="<?php esc_attr_e('Quiz name', 'studiare-core'); ?>">

                    <label for="filter_product"><?php esc_html_e('Filter by Product', 'studiare-core'); ?></label>
                    <input type="text" name="filter_product" id="filter_product" value="<?php echo isset($_GET['filter_product']) ? esc_attr(sanitize_text_field($_GET['filter_product'])) : ''; ?>" placeholder="<?php esc_attr_e('Product name', 'studiare-core'); ?>">

                    <label for="filter_user"><?php esc_html_e('Filter by User', 'studiare-core'); ?></label>
                    <input type="text" name="filter_user" id="filter_user" value="<?php echo isset($_GET['filter_user']) ? esc_attr(sanitize_text_field($_GET['filter_user'])) : ''; ?>" placeholder="<?php esc_attr_e('User name', 'studiare-core'); ?>">

                    <label for="filter_status"><?php esc_html_e('Filter by Status', 'studiare-core'); ?></label>
                    <select name="filter_status" id="filter_status">
                        <option value=""><?php esc_html_e('All', 'studiare-core'); ?></option>
                        <option value="passed" <?php selected(isset($_GET['filter_status']) && $_GET['filter_status'] === 'passed'); ?>><?php esc_html_e('Passed', 'studiare-core'); ?></option>
                        <option value="failed" <?php selected(isset($_GET['filter_status']) && $_GET['filter_status'] === 'failed'); ?>><?php esc_html_e('Failed', 'studiare-core'); ?></option>
                    </select>

                    <input type="submit" class="button" value="<?php esc_attr_e('Filter', 'studiare-core'); ?>">
                </div>
            </div>
            <?php $table->display(); ?>
        </form>
    </div>
    <?php
}
?>