<?php
/**
 * Studiare Quiz Frontend Display
 * Handles the shortcode to display quizzes in the frontend
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Check if user has purchased required products
 */
function studiare_user_has_purchased_products($user_id, $product_ids) {
    if (!function_exists('wc_customer_bought_product')) {
        return false; // WooCommerce is not active
    }

    if (empty($product_ids)) {
        return true; // No products required
    }

    if (!$user_id) {
        return false; // No user logged in
    }

    foreach ($product_ids as $product_id) {
        $product = wc_get_product($product_id);
        if ($product && wc_customer_bought_product('', $user_id, $product_id)) {
            return true; // User has purchased at least one valid product
        }
    }

    return false; // No valid products purchased
}

/**
 * Register the quiz display shortcode
 */
function studiare_quiz_shortcode($atts) {
    // Check if quiz system is enabled
    if (!get_option('studiare_quiz_post_type_enabled', true)) {
        return '<p class="studiare-quiz-message">' . esc_html__('The quiz system is currently disabled.', 'studiare-core') . '</p>';
    }
    
    // Start session if not already started
    /**if (!session_id() && !headers_sent()) {
    session_start();
    }  **/
    
   // Session handling with better error checking
    $session_status = session_status();
    
    if ($session_status === PHP_SESSION_NONE) {
        if (!headers_sent()) {
            // Try to set a custom session path if default isn't working
            $session_path = ini_get('session.save_path');
            if (empty($session_path) || !is_writable($session_path)) {
                $upload_dir = wp_upload_dir();
                $custom_path = trailingslashit($upload_dir['basedir']) . 'sessions';
                
                if (!file_exists($custom_path)) {
                    wp_mkdir_p($custom_path);
                }
                
                if (is_writable($custom_path)) {
                    ini_set('session.save_path', $custom_path);
                }
            }
            
            // Start the session with error suppression to prevent warnings
            @session_start();
        } else {
            error_log('studiare_quiz_shortcode: Cannot start session - headers already sent');
            return '<p class="studiare-quiz-message">' . esc_html__('Quiz system initialization failed.', 'studiare-core') . '</p>';
        }
    } elseif ($session_status === PHP_SESSION_DISABLED) {
        error_log('studiare_quiz_shortcode: Sessions are disabled on this server');
        return '<p class="studiare-quiz-message">' . esc_html__('Quiz system requires sessions to be enabled on the server.', 'studiare-core') . '</p>';
    }

    // Enqueue styles and scripts when shortcode is executed
    wp_enqueue_style(
        'studiare-quiz-css',
        STUDIARE_QUIZ_URL . 'assets/css/sc-quiz.css',
        array(),
        '1.0.0'
    );

    wp_enqueue_script(
        'studiare-quiz-js',
        STUDIARE_QUIZ_URL . 'assets/js/sc-quiz.js',
        array('jquery'),
        '1.0.0',
        true
    );

    // Localize script for AJAX
    wp_localize_script(
        'studiare-quiz-js',
        'studiareQuiz',
        array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'error_message' => __('Error submitting quiz.', 'studiare-core'),
            'start_error' => __('Cannot start quiz.', 'studiare-core'),
            'show_all_attempts_text' => __('Show All Attempts', 'studiare-core'),
            'hide_attempts_text' => __('Hide Attempts', 'studiare-core'),
        )
    );

    // Extract shortcode attributes
    $atts = shortcode_atts(array(
        'id' => 0,
        'pid' => null,//javad
    ), $atts, 'studiare_quiz');

    $quiz_id = intval($atts['id']);
    $pid     = $atts['pid'];//javad
    if ($quiz_id <= 0) {
        return '<p>' . esc_html__('Invalid Quiz ID', 'studiare-core') . '</p>';
    }

    // Get quiz post
    $quiz = get_post($quiz_id);
    if (!$quiz || $quiz->post_type !== 'sc-quiz') {
        return '<p>' . esc_html__('Quiz not found', 'studiare-core') . '</p>';
    }

    // Get quiz meta
    $description = get_post_meta($quiz_id, '_quiz_description', true);
    $start_time = get_post_meta($quiz_id, '_quiz_start_time', true);
    $end_time = get_post_meta($quiz_id, '_quiz_end_time', true);
    $restrict_to_members = get_post_meta($quiz_id, '_quiz_restrict_to_members', true);
    $restrict_to_products = get_post_meta($quiz_id, '_quiz_restrict_to_products', true);
    $restrict_to_products = is_array($restrict_to_products) ? $restrict_to_products : [];
    $max_attempts = intval(get_post_meta($quiz_id, '_quiz_max_attempts', true));

    // Get current time and compare with start/end times
    $current_time = current_time('timestamp');
    $start_timestamp = !empty($start_time) ? strtotime($start_time) : 0;
    $end_timestamp = !empty($end_time) ? strtotime($end_time) : PHP_INT_MAX;

    // Format start_time for JavaScript (mm/dd/YYYY HH:MM)
    $start_time_js = !empty($start_time) && $start_timestamp ? date('m/d/Y H:i', $start_timestamp) : '';

    // Check restrictions
    $user_id = get_current_user_id();
    $is_restricted_to_members = $restrict_to_members === 'yes';
    $is_restricted_to_products = !empty($restrict_to_products);
    $can_access = true;
    $restriction_message = '';

    if ($is_restricted_to_members && !$user_id) {
        $can_access = false;
        $restriction_message = __('Please log in to access this quiz.', 'studiare-core');
    }

    if ($can_access && $is_restricted_to_products && !studiare_user_has_purchased_products($user_id, $restrict_to_products)) {
        $can_access = false;
        $product_names = [];
        foreach ($restrict_to_products as $product_id) {
            $product = wc_get_product($product_id);
            if ($product) {
                $product_names[] = $product->get_name();
            }
        }
        $restriction_message = sprintf(
            __('You need to purchase one of the following products to access this quiz: %s.', 'studiare-core'),
            implode(', ', $product_names)
        );
    }

    // Check maximum attempts
    $attempt_count = 0;
    if ($can_access && $max_attempts > 0 && $user_id) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'studiare_quiz_results';
        $attempt_count = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE user_id = %d AND quiz_id = %d",
                $user_id,
                $quiz_id
            )
        );
        if ($attempt_count >= $max_attempts) {
            $can_access = false;
            $restriction_message = sprintf(
                __('You have reached the maximum number of attempts (%d) for this quiz.', 'studiare-core'),
                $max_attempts
            );
        }
    }

    // Get previous results for the current user and quiz
    $previous_results = [];
    if ($user_id) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'studiare_quiz_results';

        // Get user's quiz results for this specific quiz
        $results_raw = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT r.quiz_id, r.score, r.timestamp, r.details, p.post_title 
                 FROM $table_name r 
                 JOIN {$wpdb->posts} p ON r.quiz_id = p.ID 
                 WHERE r.user_id = %d AND r.quiz_id = %d AND p.post_type = 'sc-quiz' 
                 ORDER BY r.timestamp DESC",
                $user_id,
                $quiz_id
            )
        );

        // Process results based on score storage method
        $score_storage_method = get_post_meta($quiz_id, '_quiz_score_storage_method', true);
        $score_storage_method = in_array($score_storage_method, ['all_attempts', 'highest_score', 'average_score']) ? $score_storage_method : 'all_attempts';
        $processed_quizzes = [];

        foreach ($results_raw as $result) {
            if ($score_storage_method === 'highest_score') {
                if (!isset($processed_quizzes[$quiz_id])) {
                    // Find the attempt with the highest score
                    $highest_attempt = null;
                    $highest_score = -PHP_FLOAT_MAX;

                    foreach ($results_raw as $attempt) {
                        $details = maybe_unserialize($attempt->details);
                        $details = is_array($details) ? $details : [];
                        $is_highest = isset($details['is_highest_score']) ? $details['is_highest_score'] : null;

                        if ($is_highest === true) {
                            $highest_attempt = $attempt;
                            break;
                        }

                        // Fallback: compare scores for older data
                        $attempt_score = floatval($attempt->score);
                        if ($attempt_score > $highest_score) {
                            $highest_score = $attempt_score;
                            $highest_attempt = $attempt;
                        }
                    }

                    if ($highest_attempt) {
                        $previous_results[] = $highest_attempt;
                        $processed_quizzes[$quiz_id] = true;
                    }
                }
            } elseif ($score_storage_method === 'average_score') {
                if (!isset($processed_quizzes[$quiz_id])) {
                    // Calculate average score
                    $scores = array_map(function($attempt) {
                        return floatval($attempt->score);
                    }, $results_raw);
                    $average_score = count($scores) > 0 ? array_sum($scores) / count($scores) : 0;

                    // Calculate average percentages
                    $correct_percentages = [];
                    $incorrect_percentages = [];
                    $unanswered_percentages = [];
                    $total_possible_score = 0;

                    foreach ($results_raw as $attempt) {
                        $details = maybe_unserialize($attempt->details);
                        $details = is_array($details) ? $details : [];
                        $correct_percentages[] = isset($details['correct_percentage']) ? floatval($details['correct_percentage']) : 0;
                        $incorrect_percentages[] = isset($details['incorrect_percentage']) ? floatval($details['incorrect_percentage']) : 0;
                        $unanswered_percentages[] = isset($details['unanswered_percentage']) ? floatval($details['unanswered_percentage']) : 0;
                        if (isset($details['total_possible_score'])) {
                            $total_possible_score = floatval($details['total_possible_score']);
                        }
                    }

                    $average_correct = count($correct_percentages) > 0 ? array_sum($correct_percentages) / count($correct_percentages) : 0;
                    $average_incorrect = count($incorrect_percentages) > 0 ? array_sum($incorrect_percentages) / count($incorrect_percentages) : 0;
                    $average_unanswered = count($unanswered_percentages) > 0 ? array_sum($unanswered_percentages) / count($unanswered_percentages) : 0;

                    // Use the latest attempt for other details
                    $latest_attempt = reset($results_raw);

                    // Adjust details to reflect averages
                    $details = maybe_unserialize($latest_attempt->details);
                    $details = is_array($details) ? $details : [];
                    $details['average_score'] = $average_score;
                    $details['average_correct_percentage'] = $average_correct;
                    $details['average_incorrect_percentage'] = $average_incorrect;
                    $details['average_unanswered_percentage'] = $average_unanswered;

                    $previous_results[] = (object) [
                        'quiz_id' => $quiz_id,
                        'score' => $average_score,
                        'timestamp' => $latest_attempt->timestamp,
                        'details' => maybe_serialize($details),
                        'post_title' => $latest_attempt->post_title,
                        'attempts' => $results_raw, // Store all attempts for detailed view
                    ];

                    $processed_quizzes[$quiz_id] = true;
                }
            } else {
                // For 'all_attempts', use the raw result
                $previous_results[] = $result;
            }
        }
    }

    // Start output buffering
    ob_start();
    ?>
    <div class="studiare-quiz" pid="<?php if(!empty($pid)){echo $pid;}else{}?>" id="studiare-quiz-<?php echo esc_attr($quiz_id); ?>">
        <!-- Display previous results if user is logged in and has results -->
        <?php if ($user_id && !empty($previous_results)) : ?>
            <div class="quiz-previous-results">
                <?php
                $score_storage_method = get_post_meta($quiz_id, '_quiz_score_storage_method', true);
                $score_storage_method = in_array($score_storage_method, ['all_attempts', 'highest_score', 'average_score']) ? $score_storage_method : 'all_attempts';
                $use_accordion = count($previous_results) > 1 && $score_storage_method !== 'average_score';
                ?>

                <?php if ($use_accordion) : ?>
                    <!-- Accordion for multiple results (except average_score) -->
                    <div class="quiz-results-accordion">
                        <button class="sc-accordion-toggle" type="button">
                            <?php echo esc_html(sprintf(__('View Results (%d attempts)', 'studiare-core'), count($previous_results))); ?>
                        </button>
                        <div class="accordion-content" style="display: none;">
                            <?php foreach ($previous_results as $index => $result) : ?>
                                <?php
                                // Decode details
                                $details = maybe_unserialize($result->details);
                                $details = is_array($details) ? $details : [];
                                $status = isset($details['status']) ? $details['status'] : 'unknown';
                                $total_possible_score = isset($details['total_possible_score']) ? floatval($details['total_possible_score']) : 0;
                                $correct_percentage = isset($details['correct_percentage']) ? floatval($details['correct_percentage']) : 0;
                                $incorrect_percentage = isset($details['incorrect_percentage']) ? floatval($details['incorrect_percentage']) : 0;
                                $unanswered_percentage = isset($details['unanswered_percentage']) ? floatval($details['unanswered_percentage']) : 0;

                                // Convert timestamp to local time
                                $timestamp = strtotime($result->timestamp . ' UTC');
                                $local_time = date_i18n(
                                    get_option('date_format') . ' ' . get_option('time_format'),
                                    $timestamp
                                );
                                ?>
                                <div class="quiz-result-entry">
                                    <p><?php echo esc_html(sprintf(__('Score: %s out of %s', 'studiare-core'), number_format($result->score, 1), $total_possible_score)); ?></p>
                                    <p>
                                        <?php esc_html_e('Status:', 'studiare-core'); ?> 
                                        <span class="<?php echo esc_attr($status === 'passed' ? 'status-passed' : 'status-failed'); ?>">
                                            <?php echo esc_html($status === 'passed' ? __('Passed', 'studiare-core') : __('Failed', 'studiare-core')); ?>
                                        </span>
                                    </p>
                                    <table class="quiz-stats-table">
                                        <tr>
                                            <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($correct_percentage, 1))); ?></td>
                                            <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($incorrect_percentage, 1))); ?></td>
                                            <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($unanswered_percentage, 1))); ?></td>
                                        </tr>
                                        <tr>
                                            <td><?php esc_html_e('Correct', 'studiare-core'); ?></td>
                                            <td><?php esc_html_e('Incorrect', 'studiare-core'); ?></td>
                                            <td><?php esc_html_e('Unanswered', 'studiare-core'); ?></td>
                                        </tr>
                                    </table>
                                    <p><?php echo esc_html(sprintf(__('Date: %s', 'studiare-core'), $local_time)); ?></p>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php else : ?>
                    <!-- Direct display for single result or average_score -->
                <?php foreach ($previous_results as $index => $result) : ?>
                    <?php
                    // Decode details
                    $details = maybe_unserialize($result->details);
                    $details = is_array($details) ? $details : [];
                    $status = isset($details['status']) ? $details['status'] : 'unknown';
                    $total_possible_score = isset($details['total_possible_score']) ? floatval($details['total_possible_score']) : 0;
                    $correct_percentage = isset($details['correct_percentage']) ? floatval($details['correct_percentage']) : 0;
                    $incorrect_percentage = isset($details['incorrect_percentage']) ? floatval($details['incorrect_percentage']) : 0;
                    $unanswered_percentage = isset($details['unanswered_percentage']) ? floatval($details['unanswered_percentage']) : 0;
                    $average_score = isset($details['average_score']) ? floatval($details['average_score']) : null;
                    $average_correct = isset($details['average_correct_percentage']) ? floatval($details['average_correct_percentage']) : null;
                    $average_incorrect = isset($details['average_incorrect_percentage']) ? floatval($details['average_incorrect_percentage']) : null;
                    $average_unanswered = isset($details['average_unanswered_percentage']) ? floatval($details['average_unanswered_percentage']) : null;
                    $attempts = isset($result->attempts) ? $result->attempts : [];

                    // Unique ID for each result
                    $unique_id = $result->quiz_id . '-' . $user_id . '-' . $index;

                    // Convert timestamp to local time
                    $timestamp = strtotime($result->timestamp . ' UTC');
                    $local_time = date_i18n(
                        get_option('date_format') . ' ' . get_option('time_format'),
                        $timestamp
                    );

                    // Determine displayed score
                    $display_score = $average_score !== null ? $average_score : $result->score;
                    ?>
                    <div class="quiz-result-entry">
                        <?php if ($average_score !== null) : ?>
                            <!-- Display average stats for average_score mode -->
                            <p><?php echo esc_html(sprintf(__('Average Score: %s out of %s', 'studiare-core'), number_format($average_score, 1), $total_possible_score)); ?></p>
                            <table class="quiz-stats-table">
                                <tr>
                                    <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($average_correct, 1))); ?></td>
                                    <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($average_incorrect, 1))); ?></td>
                                    <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($average_unanswered, 1))); ?></td>
                                </tr>
                                <tr>
                                    <td><?php esc_html_e('Average Correct', 'studiare-core'); ?></td>
                                    <td><?php esc_html_e('Average Incorrect', 'studiare-core'); ?></td>
                                    <td><?php esc_html_e('Average Unanswered', 'studiare-core'); ?></td>
                                </tr>
                            </table>
                            <!-- Show All Attempts button for average_score mode -->
                            <?php if (!empty($attempts)) : ?>
                                <button class="button show-all-attempts" data-attempts-id="<?php echo esc_attr($unique_id); ?>">
                                    <?php esc_html_e('Show All Attempts', 'studiare-core'); ?>
                                </button>
                                <div class="quiz-attempts-details" id="attempts-details-<?php echo esc_attr($unique_id); ?>" style="display: none;">
                                    <h4><?php esc_html_e('All Attempts', 'studiare-core'); ?></h4>
                                    <?php
                                    $total_attempts = count($attempts);
                                    $display_number = $total_attempts;
                                    ?>
                                    <?php foreach ($attempts as $attempt) : ?>
                                        <?php
                                        $attempt_details = maybe_unserialize($attempt->details);
                                        $attempt_details = is_array($attempt_details) ? $attempt_details : [];
                                        $attempt_score = floatval($attempt->score);
                                        $attempt_correct = isset($attempt_details['correct_percentage']) ? floatval($attempt_details['correct_percentage']) : 0;
                                        $attempt_incorrect = isset($attempt_details['incorrect_percentage']) ? floatval($attempt_details['incorrect_percentage']) : 0;
                                        $attempt_unanswered = isset($attempt_details['unanswered_percentage']) ? floatval($attempt_details['unanswered_percentage']) : 0;
                                        $attempt_timestamp = strtotime($attempt->timestamp . ' UTC');
                                        $attempt_local_time = date_i18n(
                                            get_option('date_format') . ' ' . get_option('time_format'),
                                            $attempt_timestamp
                                        );
                                        $attempt_status = isset($attempt_details['status']) ? $attempt_details['status'] : 'unknown';
                                        ?>
                                        <div class="quiz-attempt">
                                            <p><?php echo esc_html(sprintf(__('Attempt %d:', 'studiare-core'), $display_number)); ?></p>
                                            <p><?php echo esc_html(sprintf(__('Score: %s out of %s', 'studiare-core'), number_format($attempt_score, 1), $total_possible_score)); ?></p>
                                            <p>
                                                <?php esc_html_e('Status:', 'studiare-core'); ?> 
                                                <span class="<?php echo esc_attr($attempt_status === 'passed' ? 'status-passed' : 'status-failed'); ?>">
                                                    <?php echo esc_html($attempt_status === 'passed' ? __('Passed', 'studiare-core') : __('Failed', 'studiare-core')); ?>
                                                </span>
                                            </p>
                                            <table class="quiz-stats-table">
                                                <tr>
                                                    <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($attempt_correct, 1))); ?></td>
                                                    <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($attempt_incorrect, 1))); ?></td>
                                                    <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($attempt_unanswered, 1))); ?></td>
                                                </tr>
                                                <tr>
                                                    <td><?php esc_html_e('Correct', 'studiare-core'); ?></td>
                                                    <td><?php esc_html_e('Incorrect', 'studiare-core'); ?></td>
                                                    <td><?php esc_html_e('Unanswered', 'studiare-core'); ?></td>
                                                </tr>
                                            </table>
                                            <p><?php echo esc_html(sprintf(__('Date: %s', 'studiare-core'), $attempt_local_time)); ?></p>
                                        </div>
                                        <?php $display_number--; ?>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                        <?php else : ?>
                            <!-- Display stats for all_attempts or highest_score modes -->
                            <p><?php echo esc_html(sprintf(__('Score: %s out of %s', 'studiare-core'), number_format($result->score, 1), $total_possible_score)); ?></p>
                            <p>
                                <?php esc_html_e('Status:', 'studiare-core'); ?> 
                                <span class="<?php echo esc_attr($status === 'passed' ? 'status-passed' : 'status-failed'); ?>">
                                    <?php echo esc_html($status === 'passed' ? __('Passed', 'studiare-core') : __('Failed', 'studiare-core')); ?>
                                </span>
                            </p>
                            <table class="quiz-stats-table">
                                <tr>
                                    <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($correct_percentage, 1))); ?></td>
                                    <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($incorrect_percentage, 1))); ?></td>
                                    <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($unanswered_percentage, 1))); ?></td>
                                </tr>
                                <tr>
                                    <td><?php esc_html_e('Correct', 'studiare-core'); ?></td>
                                    <td><?php esc_html_e('Incorrect', 'studiare-core'); ?></td>
                                    <td><?php esc_html_e('Unanswered', 'studiare-core'); ?></td>
                                </tr>
                            </table>
                            <p><?php echo esc_html(sprintf(__('Date: %s', 'studiare-core'), $local_time)); ?></p>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
                <?php endif; ?>
            </div>
        <?php elseif ($user_id) : ?>
            <!-- No previous results -->
            <div class="quiz-previous-results">
                <p><?php esc_html_e('You have not attempted this quiz yet.', 'studiare-core'); ?></p>
            </div>
        <?php endif; ?>
        <h2><?php echo esc_html($quiz->post_title); ?></h2>
        <?php if (!empty($description)) : ?>
            <div class="quiz-description">
                <?php echo wp_kses_post($description); ?>
            </div>
        <?php endif; ?>
        

        <?php if (!$can_access) : ?>
            <!-- Restriction message -->
            <div class="quiz-restricted">
                <p><?php echo esc_html($restriction_message); ?></p>
                <?php if ($is_restricted_to_members && !$user_id) : ?>
                    <p>
                        <a href="<?php echo esc_url(function_exists('wc_get_page_permalink') ? wc_get_page_permalink('myaccount') : wp_login_url(get_permalink())); ?>" class="quiz-login-link">
                            <?php esc_html_e('Log in', 'studiare-core'); ?>
                        </a>
                    </p>
                <?php endif; ?>
                <?php if ($is_restricted_to_products && !studiare_user_has_purchased_products($user_id, $restrict_to_products)) : ?>
                    <p>
                        <?php foreach ($restrict_to_products as $product_id) : ?>
                            <?php $product = wc_get_product($product_id); ?>
                            <?php if ($product) : ?>
                                <a href="<?php echo esc_url($product->get_permalink()); ?>" class="quiz-product-link"><?php echo esc_html(sprintf(__('Buy %s', 'studiare-core'), $product->get_name())); ?></a><br>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </p>
                <?php endif; ?>
            </div>
        <?php elseif ($start_timestamp && $current_time < $start_timestamp) : ?>
            <!-- Before start time: show countdown to start -->
            <div class="quiz-pre-start">
                <p><?php esc_html_e('This quiz will start in:', 'studiare-core'); ?></p>
                <div class="quiz-start-countdown" id="quiz-start-countdown-<?php echo esc_attr($quiz_id); ?>" data-start-time="<?php echo esc_attr($start_time_js); ?>">
                    <span class="countdown-display">
                        <span class="countdown-days"></span> <?php esc_html_e('days', 'studiare-core'); ?>
                        <span class="countdown-hours"></span> <?php esc_html_e('hours', 'studiare-core'); ?>
                        <span class="countdown-minutes"></span> <?php esc_html_e('minutes', 'studiare-core'); ?>
                        <span class="countdown-seconds"></span> <?php esc_html_e('seconds', 'studiare-core'); ?>
                    </span>
                </div>
            </div>
        <?php elseif ($end_timestamp && $current_time > $end_timestamp) : ?>
            <!-- After end time: show end message -->
            <div class="quiz-ended">
                <p>
                    <?php
                    printf(
                        esc_html__('This quiz ended on %s.', 'studiare-core'),
                        esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $end_timestamp))
                    );
                    ?>
                </p>
            </div>
        <?php else : ?>
            <!-- Within time range or no restrictions: show start button -->
            <?php if (isset($_SESSION['quiz_result_' . $quiz_id])) : ?>
                <div class="quiz-result">
                    <h3><?php echo esc_html(sprintf(__('Your Score: %s', 'studiare-core'), $_SESSION['quiz_result_' . $quiz_id]['score'])); ?></h3>
                    <p><?php echo esc_html(sprintf(__('Correct: %s%%', 'studiare-core'), number_format($_SESSION['quiz_result_' . $quiz_id]['correct_percentage'], 1))); ?></p>
                    <p><?php echo esc_html(sprintf(__('Incorrect: %s%%', 'studiare-core'), number_format($_SESSION['quiz_result_' . $quiz_id]['incorrect_percentage'], 1))); ?></p>
                    <p><?php echo esc_html(sprintf(__('Unanswered: %s%%', 'studiare-core'), number_format($_SESSION['quiz_result_' . $quiz_id]['unanswered_percentage'], 1))); ?></p>
                    <?php if (isset($_SESSION['quiz_result_' . $quiz_id]['status'])) : ?>
                        <p>
                            <?php esc_html_e('Status:', 'studiare-core'); ?> 
                            <span class="<?php echo esc_attr($_SESSION['quiz_result_' . $quiz_id]['status'] === 'passed' ? 'status-passed' : 'status-failed'); ?>">
                                <?php echo esc_html($_SESSION['quiz_result_' . $quiz_id]['status'] === 'passed' ? __('Passed', 'studiare-core') : __('Failed', 'studiare-core')); ?>
                            </span>
                        </p>
                    <?php endif; ?>
                </div>
                <?php unset($_SESSION['quiz_result_' . $quiz_id]); ?>
            <?php else : ?>
                <div class="quiz-start">
                    <button type="button" class="start-quiz-button" data-quiz-id="<?php echo esc_attr($quiz_id); ?>">
                        <?php esc_html_e('Start Quiz', 'studiare-core'); ?>
                    </button>
                </div>
                <div class="quiz-content" id="quiz-content-<?php echo esc_attr($quiz_id); ?>" style="display: none;"></div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
    <?php
    $output = ob_get_clean();
    return $output;
}
add_shortcode('studiare_quiz', 'studiare_quiz_shortcode');

/**
 * Register the quiz results shortcode
 */
function studiare_quiz_results_shortcode() {
    // Check if quiz system is enabled
    if (!get_option('studiare_quiz_post_type_enabled', true)) {
        return '<p class="studiare-quiz-message">' . esc_html__('The quiz system is currently disabled.', 'studiare-core') . '</p>';
    }

    // Enqueue styles and scripts if not already enqueued
    if (!wp_style_is('studiare-quiz-css', 'enqueued')) {
        wp_enqueue_style(
            'studiare-quiz-css',
            STUDIARE_QUIZ_URL . 'assets/css/sc-quiz.css',
            array(),
            '1.0.0'
        );
    }

    if (!wp_script_is('studiare-quiz-js', 'enqueued')) {
        wp_enqueue_script(
            'studiare-quiz-js',
            STUDIARE_QUIZ_URL . 'assets/js/sc-quiz.js',
            array('jquery'),
            '1.0.0',
            true
        );

        // Localize script for AJAX
        wp_localize_script(
            'studiare-quiz-js',
            'studiareQuiz',
            array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'error_message' => __('Error submitting quiz.', 'studiare-core'),
                'start_error' => __('Cannot start quiz.', 'studiare-core'),
                'show_all_attempts_text' => __('Show All Attempts', 'studiare-core'),
                'hide_attempts_text' => __('Hide Attempts', 'studiare-core'),
            )
        );
    }

    if (!is_user_logged_in()) {
        return '<p class="studiare-quiz-message">' . esc_html__('Please log in to view your quiz results.', 'studiare-core') . '</p>';
    }

    $user_id = get_current_user_id();
    global $wpdb;
    $table_name = $wpdb->prefix . 'studiare_quiz_results';

    // Get user's quiz results with product information
    $results_raw = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT r.quiz_id, r.score, r.timestamp, r.details, r.product_id, p.post_title AS quiz_title, prod.post_title AS product_title
             FROM $table_name r 
             JOIN {$wpdb->posts} p ON r.quiz_id = p.ID 
             LEFT JOIN {$wpdb->posts} prod ON r.product_id = prod.ID
             WHERE r.user_id = %d AND p.post_type = 'sc-quiz'
             ORDER BY r.quiz_id, r.timestamp DESC",
            $user_id
        )
    );

    if (empty($results_raw)) {
        return '<p class="studiare-quiz-message">' . esc_html__('You have not participated in any quizzes yet.', 'studiare-core') . '</p>';
    }

    // Process results based on score storage method
    $results = [];
    $processed_quizzes = [];

    foreach ($results_raw as $result) {
        $quiz_id = $result->quiz_id;
        $score_storage_method = get_post_meta($quiz_id, '_quiz_score_storage_method', true);
        $score_storage_method = in_array($score_storage_method, ['all_attempts', 'highest_score', 'average_score']) ? $score_storage_method : 'all_attempts';

        // Ensure quiz_title is set, fallback to a default if empty
        $result->quiz_title = !empty($result->quiz_title) ? $result->quiz_title : __('Untitled Quiz', 'studiare-core');

        if ($score_storage_method === 'highest_score') {
            if (!isset($processed_quizzes[$quiz_id])) {
                // Get all attempts for this quiz
                $quiz_attempts = array_filter($results_raw, function($r) use ($quiz_id) {
                    return $r->quiz_id == $quiz_id;
                });

                if (!empty($quiz_attempts)) {
                    // Find the attempt with the highest score
                    $highest_attempt = null;
                    $highest_score = -PHP_FLOAT_MAX;

                    foreach ($quiz_attempts as $attempt) {
                        $details = maybe_unserialize($attempt->details);
                        $details = is_array($details) ? $details : [];
                        $is_highest = isset($details['is_highest_score']) ? $details['is_highest_score'] : null;

                        if ($is_highest === true) {
                            $highest_attempt = $attempt;
                            break;
                        }

                        // Fallback: if is_highest_score flag is not set (old data), compare scores
                        $attempt_score = floatval($attempt->score);
                        if ($attempt_score > $highest_score) {
                            $highest_score = $attempt_score;
                            $highest_attempt = $attempt;
                        }
                    }

                    if ($highest_attempt) {
                        // Ensure quiz_title is preserved
                        $highest_attempt->quiz_title = !empty($highest_attempt->quiz_title) ? $highest_attempt->quiz_title : __('Untitled Quiz', 'studiare-core');
                        $results[] = $highest_attempt;
                        $processed_quizzes[$quiz_id] = true;
                    }
                }
            }
        } elseif ($score_storage_method === 'average_score') {
            if (!isset($processed_quizzes[$quiz_id])) {
                // Get all attempts for this quiz
                $quiz_attempts = array_filter($results_raw, function($r) use ($quiz_id) {
                    return $r->quiz_id == $quiz_id;
                });

                if (!empty($quiz_attempts)) {
                    // Calculate average score
                    $scores = array_map(function($attempt) {
                        return floatval($attempt->score);
                    }, $quiz_attempts);
                    $average_score = array_sum($scores) / count($scores);

                    // Calculate average percentages for Correct, Incorrect, Unanswered
                    $correct_percentages = [];
                    $incorrect_percentages = [];
                    $unanswered_percentages = [];
                    $total_possible_score = 0;

                    foreach ($quiz_attempts as $attempt) {
                        $details = maybe_unserialize($attempt->details);
                        $details = is_array($details) ? $details : [];
                        $correct_percentages[] = isset($details['correct_percentage']) ? floatval($details['correct_percentage']) : 0;
                        $incorrect_percentages[] = isset($details['incorrect_percentage']) ? floatval($details['incorrect_percentage']) : 0;
                        $unanswered_percentages[] = isset($details['unanswered_percentage']) ? floatval($details['unanswered_percentage']) : 0;
                        if (isset($details['total_possible_score'])) {
                            $total_possible_score = floatval($details['total_possible_score']);
                        }
                    }

                    $average_correct = count($correct_percentages) > 0 ? array_sum($correct_percentages) / count($correct_percentages) : 0;
                    $average_incorrect = count($incorrect_percentages) > 0 ? array_sum($incorrect_percentages) / count($incorrect_percentages) : 0;
                    $average_unanswered = count($unanswered_percentages) > 0 ? array_sum($unanswered_percentages) / count($unanswered_percentages) : 0;

                    // Use the latest attempt for other details
                    $latest_attempt = reset($quiz_attempts);

                    // Adjust details to reflect averages
                    $details = maybe_unserialize($latest_attempt->details);
                    $details = is_array($details) ? $details : [];
                    $details['average_score'] = $average_score;
                    $details['average_correct_percentage'] = $average_correct;
                    $details['average_incorrect_percentage'] = $average_incorrect;
                    $details['average_unanswered_percentage'] = $average_unanswered;

                    $results[] = (object) [
                        'quiz_id' => $quiz_id,
                        'score' => $average_score,
                        'timestamp' => $latest_attempt->timestamp,
                        'details' => maybe_serialize($details),
                        'quiz_title' => !empty($latest_attempt->quiz_title) ? $latest_attempt->quiz_title : __('Untitled Quiz', 'studiare-core'),
                        'product_id' => $latest_attempt->product_id,
                        'product_title' => $latest_attempt->product_title,
                        'attempts' => $quiz_attempts, // Store all attempts for detailed view
                    ];

                    $processed_quizzes[$quiz_id] = true;
                }
            }
        } else {
            // For 'all_attempts', use the raw result
            $results[] = $result;
        }
    }

    if (empty($results)) {
        return '<p class="studiare-quiz-message">' . esc_html__('You have not participated in any quizzes yet.', 'studiare-core') . '</p>';
    }

    ob_start();
    ?>
    <div class="studiare-quiz-results" data-user-id="<?php echo esc_attr($user_id); ?>">
        <h2><?php esc_html_e('My Quiz Results', 'studiare-core'); ?></h2>
        <table class="studiare-quiz-results-table">
            <thead>
                <tr>
                    <th><?php esc_html_e('Quiz', 'studiare-core'); ?></th>
                    <th><?php esc_html_e('Product/Course', 'studiare-core'); ?></th>
                    <th><?php esc_html_e('Score', 'studiare-core'); ?></th>
                    <th><?php esc_html_e('Date', 'studiare-core'); ?></th>
                    <th><?php esc_html_e('Status', 'studiare-core'); ?></th>
                    <th><?php esc_html_e('Details', 'studiare-core'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($results as $index => $result) : ?>
                    <?php
                    // Decode details
                    $details = maybe_unserialize($result->details);
                    $details = is_array($details) ? $details : [];
                    $status = isset($details['status']) ? $details['status'] : 'unknown';

                    // Unique ID for each row
                    $unique_id = $result->quiz_id . '-' . $user_id . '-' . $index;

                    // Convert timestamp to local time using WordPress timezone
                    $timestamp = strtotime($result->timestamp . ' UTC');
                    $local_time = date_i18n(
                        get_option('date_format') . ' ' . get_option('time_format'),
                        $timestamp
                    );

                    // Determine displayed score
                    $average_score = isset($details['average_score']) ? floatval($details['average_score']) : null;
                    $display_score = $average_score !== null ? $average_score : $result->score;

                    // Get product title, fallback if not set
                    $product_title = !empty($result->product_title) ? $result->product_title : __('N/A', 'studiare-core');
                    ?>
                    <tr>
                        <td><?php echo esc_html($result->quiz_title); ?></td>
                        <td><?php echo esc_html($product_title); ?></td>
                        <td><?php echo esc_html(number_format($display_score, 1)); ?></td>
                        <td><?php echo esc_html($local_time); ?></td>
                        <td class="quiz-status <?php echo esc_attr($status === 'passed' ? 'status-passed' : 'status-failed'); ?>">
                            <?php echo esc_html($status === 'passed' ? __('Passed', 'studiare-core') : __('Failed', 'studiare-core')); ?>
                        </td>
                        <td>
                            <button class="button quiz-result-details" data-result-id="<?php echo esc_attr($unique_id); ?>">
                                <?php esc_html_e('View Details', 'studiare-core'); ?>
                            </button>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <!-- Modal for displaying quiz result details -->
        <div id="quiz-result-details-modal" class="quiz-modal" style="display: none;">
            <div class="quiz-modal-content">
                <span class="quiz-modal-close">×</span>
                <div id="quiz-result-details-content"></div>
            </div>
        </div>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode('studiare_quiz_results', 'studiare_quiz_results_shortcode');
/**
 * AJAX handler to load quiz result details
 */
function studiare_load_quiz_result_details_ajax() {
    // Check if quiz system is enabled
    if (!get_option('studiare_quiz_post_type_enabled', true)) {
        wp_send_json_error(['message' => __('The quiz system is currently disabled.', 'studiare-core')]);
    }

    if (!isset($_POST['result_id']) || !isset($_POST['user_id'])) {
        wp_send_json_error(['message' => __('Invalid request.', 'studiare-core')]);
    }

    $result_id = sanitize_text_field($_POST['result_id']);
    $user_id = intval($_POST['user_id']);
    $current_user_id = get_current_user_id();

    // Security check: ensure the user can access this result
    if ($user_id !== $current_user_id) {
        wp_send_json_error(['message' => __('You do not have permission to view this result.', 'studiare-core')]);
    }

    // Parse the result_id (format: quiz_id-user_id-index)
    $parts = explode('-', $result_id);
    if (count($parts) !== 3) {
        wp_send_json_error(['message' => __('Invalid result ID.', 'studiare-core')]);
    }

    $quiz_id = intval($parts[0]);
    $index = intval($parts[2]);

    global $wpdb;
    $table_name = $wpdb->prefix . 'studiare_quiz_results';

    // Get user's quiz results
    $results_raw = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT r.quiz_id, r.score, r.timestamp, r.details, p.post_title 
             FROM $table_name r 
             JOIN {$wpdb->posts} p ON r.quiz_id = p.ID 
             WHERE r.user_id = %d AND p.post_type = 'sc-quiz' 
             ORDER BY r.quiz_id, r.timestamp DESC",
            $user_id
        )
    );

    if (empty($results_raw)) {
        wp_send_json_error(['message' => __('No results found.', 'studiare-core')]);
    }

    // Process results based on score storage method
    $results = [];
    $processed_quizzes = [];

    foreach ($results_raw as $result) {
        $quiz_id_loop = $result->quiz_id;
        $score_storage_method = get_post_meta($quiz_id_loop, '_quiz_score_storage_method', true);
        $score_storage_method = in_array($score_storage_method, ['all_attempts', 'highest_score', 'average_score']) ? $score_storage_method : 'all_attempts';

        if ($score_storage_method === 'highest_score') {
            if (!isset($processed_quizzes[$quiz_id_loop])) {
                $quiz_attempts = array_filter($results_raw, function($r) use ($quiz_id_loop) {
                    return $r->quiz_id == $quiz_id_loop;
                });

                if (!empty($quiz_attempts)) {
                    $highest_attempt = null;
                    $highest_score = -PHP_FLOAT_MAX;

                    foreach ($quiz_attempts as $attempt) {
                        $details = maybe_unserialize($attempt->details);
                        $details = is_array($details) ? $details : [];
                        $is_highest = isset($details['is_highest_score']) ? $details['is_highest_score'] : null;

                        if ($is_highest === true) {
                            $highest_attempt = $attempt;
                            break;
                        }

                        $attempt_score = floatval($attempt->score);
                        if ($attempt_score > $highest_score) {
                            $highest_score = $attempt_score;
                            $highest_attempt = $attempt;
                        }
                    }

                    if ($highest_attempt) {
                        $results[] = $highest_attempt;
                        $processed_quizzes[$quiz_id_loop] = true;
                    }
                }
            }
        } elseif ($score_storage_method === 'average_score') {
            if (!isset($processed_quizzes[$quiz_id_loop])) {
                $quiz_attempts = array_filter($results_raw, function($r) use ($quiz_id_loop) {
                    return $r->quiz_id == $quiz_id_loop;
                });

                if (!empty($quiz_attempts)) {
                    $scores = array_map(function($attempt) {
                        return floatval($attempt->score);
                    }, $quiz_attempts);
                    $average_score = array_sum($scores) / count($scores);

                    $correct_percentages = [];
                    $incorrect_percentages = [];
                    $unanswered_percentages = [];
                    $total_possible_score = 0;

                    foreach ($quiz_attempts as $attempt) {
                        $details = maybe_unserialize($attempt->details);
                        $details = is_array($details) ? $details : [];
                        $correct_percentages[] = isset($details['correct_percentage']) ? floatval($details['correct_percentage']) : 0;
                        $incorrect_percentages[] = isset($details['incorrect_percentage']) ? floatval($details['incorrect_percentage']) : 0;
                        $unanswered_percentages[] = isset($details['unanswered_percentage']) ? floatval($details['unanswered_percentage']) : 0;
                        if (isset($details['total_possible_score'])) {
                            $total_possible_score = floatval($details['total_possible_score']);
                        }
                    }

                    $average_correct = count($correct_percentages) > 0 ? array_sum($correct_percentages) / count($correct_percentages) : 0;
                    $average_incorrect = count($incorrect_percentages) > 0 ? array_sum($incorrect_percentages) / count($incorrect_percentages) : 0;
                    $average_unanswered = count($unanswered_percentages) > 0 ? array_sum($unanswered_percentages) / count($unanswered_percentages) : 0;

                    $latest_attempt = reset($quiz_attempts);
                    $details = maybe_unserialize($latest_attempt->details);
                    $details = is_array($details) ? $details : [];
                    $details['average_score'] = $average_score;
                    $details['average_correct_percentage'] = $average_correct;
                    $details['average_incorrect_percentage'] = $average_incorrect;
                    $details['average_unanswered_percentage'] = $average_unanswered;

                    $results[] = (object) [
                        'quiz_id' => $quiz_id_loop,
                        'score' => $average_score,
                        'timestamp' => $latest_attempt->timestamp,
                        'details' => maybe_serialize($details),
                        'post_title' => $latest_attempt->post_title,
                        'attempts' => $quiz_attempts,
                    ];

                    $processed_quizzes[$quiz_id_loop] = true;
                }
            }
        } else {
            $results[] = $result;
        }
    }

    // Check if the requested index exists
    if (!isset($results[$index])) {
        wp_send_json_error(['message' => __('Result not found.', 'studiare-core')]);
    }

    $result = $results[$index];
    if ($result->quiz_id != $quiz_id) {
        wp_send_json_error(['message' => __('Result mismatch.', 'studiare-core')]);
    }

    // Decode details
    $details = maybe_unserialize($result->details);
    $details = is_array($details) ? $details : [];
    $total_possible_score = isset($details['total_possible_score']) ? floatval($details['total_possible_score']) : 0;
    $correct_percentage = isset($details['correct_percentage']) ? floatval($details['correct_percentage']) : 0;
    $incorrect_percentage = isset($details['incorrect_percentage']) ? floatval($details['incorrect_percentage']) : 0;
    $unanswered_percentage = isset($details['unanswered_percentage']) ? floatval($details['unanswered_percentage']) : 0;
    $average_score = isset($details['average_score']) ? floatval($details['average_score']) : null;
    $average_correct = isset($details['average_correct_percentage']) ? floatval($details['average_correct_percentage']) : null;
    $average_incorrect = isset($details['average_incorrect_percentage']) ? floatval($details['average_incorrect_percentage']) : null;
    $average_unanswered = isset($details['average_unanswered_percentage']) ? floatval($details['average_unanswered_percentage']) : null;
    $attempts = isset($result->attempts) ? $result->attempts : [];
    $status = isset($details['status']) ? $details['status'] : 'unknown';

    // Generate HTML for the modal content
    ob_start();
    ?>
    <h3><?php echo esc_html($result->post_title); ?></h3>
    <?php if ($average_score !== null): ?>
        <!-- Display average stats for average_score mode -->
        <p><?php echo esc_html(sprintf(__('Average Score: %s out of %s', 'studiare-core'), number_format($average_score, 1), $total_possible_score)); ?></p>
        <table class="quiz-stats-table">
            <tr>
                <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($average_correct, 1))); ?></td>
                <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($average_incorrect, 1))); ?></td>
                <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($average_unanswered, 1))); ?></td>
            </tr>
            <tr>
                <td><?php esc_html_e('Average Correct', 'studiare-core'); ?></td>
                <td><?php esc_html_e('Average Incorrect', 'studiare-core'); ?></td>
                <td><?php esc_html_e('Average Unanswered', 'studiare-core'); ?></td>
            </tr>
        </table>

        <!-- Show All Attempts button -->
        <?php if (!empty($attempts)): ?>
            <button class="button show-all-attempts" data-attempts-id="<?php echo esc_attr($result_id); ?>">
                <?php esc_html_e('Show All Attempts', 'studiare-core'); ?>
            </button>
            <div class="quiz-attempts-details" id="attempts-details-<?php echo esc_attr($result_id); ?>" style="display: none;">
                <h4><?php esc_html_e('All Attempts', 'studiare-core'); ?></h4>
                <?php
                $total_attempts = count($attempts);
                $display_number = $total_attempts;
                ?>
                <?php foreach ($attempts as $attempt): ?>
                    <?php
                    $attempt_details = maybe_unserialize($attempt->details);
                    $attempt_details = is_array($attempt_details) ? $attempt_details : [];
                    $attempt_score = floatval($attempt->score);
                    $attempt_correct = isset($attempt_details['correct_percentage']) ? floatval($attempt_details['correct_percentage']) : 0;
                    $attempt_incorrect = isset($attempt_details['incorrect_percentage']) ? floatval($attempt_details['incorrect_percentage']) : 0;
                    $attempt_unanswered = isset($attempt_details['unanswered_percentage']) ? floatval($attempt_details['unanswered_percentage']) : 0;
                    $attempt_timestamp = strtotime($attempt->timestamp . ' UTC');
                    $attempt_local_time = date_i18n(
                        get_option('date_format') . ' ' . get_option('time_format'),
                        $attempt_timestamp
                    );
                    $attempt_status = isset($attempt_details['status']) ? $attempt_details['status'] : 'unknown';
                    ?>
                    <div class="quiz-attempt">
                        <p><?php echo esc_html(sprintf(__('Attempt %d:', 'studiare-core'), $display_number)); ?></p>
                        <p><?php echo esc_html(sprintf(__('Score: %s out of %s', 'studiare-core'), number_format($attempt_score, 1), $total_possible_score)); ?></p>
                        <p>
                            <?php esc_html_e('Status:', 'studiare-core'); ?> 
                            <span class="<?php echo esc_attr($attempt_status === 'passed' ? 'status-passed' : 'status-failed'); ?>">
                                <?php echo esc_html($attempt_status === 'passed' ? __('Passed', 'studiare-core') : __('Failed', 'studiare-core')); ?>
                            </span>
                        </p>
                        <table class="quiz-stats-table">
                            <tr>
                                <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($attempt_correct, 1))); ?></td>
                                <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($attempt_incorrect, 1))); ?></td>
                                <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($attempt_unanswered, 1))); ?></td>
                            </tr>
                            <tr>
                                <td><?php esc_html_e('Correct', 'studiare-core'); ?></td>
                                <td><?php esc_html_e('Incorrect', 'studiare-core'); ?></td>
                                <td><?php esc_html_e('Unanswered', 'studiare-core'); ?></td>
                            </tr>
                        </table>
                        <p><?php echo esc_html(sprintf(__('Date: %s', 'studiare-core'), $attempt_local_time)); ?></p>
                    </div>
                    <?php $display_number--; ?>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    <?php else: ?>
        <!-- Display regular stats for all_attempts or highest_score modes -->
        <p><?php echo esc_html(sprintf(__('Your Score: %s out of %s', 'studiare-core'), number_format($result->score, 1), $total_possible_score)); ?></p>
        <p>
            <?php esc_html_e('Status:', 'studiare-core'); ?> 
            <span class="<?php echo esc_attr($status === 'passed' ? 'status-passed' : 'status-failed'); ?>">
                <?php echo esc_html($status === 'passed' ? __('Passed', 'studiare-core') : __('Failed', 'studiare-core')); ?>
            </span>
        </p>
        <table class="quiz-stats-table">
            <tr>
                <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($correct_percentage, 1))); ?></td>
                <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($incorrect_percentage, 1))); ?></td>
                <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($unanswered_percentage, 1))); ?></td>
            </tr>
            <tr>
                <td><?php esc_html_e('Correct', 'studiare-core'); ?></td>
                <td><?php esc_html_e('Incorrect', 'studiare-core'); ?></td>
                <td><?php esc_html_e('Unanswered', 'studiare-core'); ?></td>
            </tr>
        </table>
    <?php endif; ?>
    <?php
    $html = ob_get_clean();
    wp_send_json_success(['html' => $html]);
}
add_action('wp_ajax_studiare_load_quiz_result_details', 'studiare_load_quiz_result_details_ajax');

/**
 * AJAX handler for loading quiz content
 */
function studiare_load_quiz_content_ajax() {
    // Check if quiz system is enabled
    if (!get_option('studiare_quiz_post_type_enabled', true)) {
        wp_send_json_error(['message' => __('The quiz system is currently disabled.', 'studiare-core')]);
    }

    // Start session if not already started
    if (!session_id() && !headers_sent()) {
        session_start();
    }

    $quiz_id = isset($_POST['quiz_id']) ? intval($_POST['quiz_id']) : 0;
    $pid = isset($_POST['pid']) ? intval($_POST['pid']) : 0;//javad
    if ($quiz_id <= 0) {
        wp_send_json_error(['message' => __('Invalid quiz ID.', 'studiare-core')]);
    }

    // Get quiz post
    $quiz = get_post($quiz_id);
    if (!$quiz || $quiz->post_type !== 'sc-quiz') {
        wp_send_json_error(['message' => __('Quiz not found.', 'studiare-core')]);
    }

    // Get quiz meta
    $start_time = get_post_meta($quiz_id, '_quiz_start_time', true);
    $end_time = get_post_meta($quiz_id, '_quiz_end_time', true);
    $answer_time = get_post_meta($quiz_id, '_quiz_answer_time', true);
    $questions = get_post_meta($quiz_id, '_quiz_questions', true);
    $enable_pagination = get_post_meta($quiz_id, '_quiz_enable_pagination', true) === 'yes';
    $questions_per_page = intval(get_post_meta($quiz_id, '_quiz_questions_per_page', true));
    $restrict_to_members = get_post_meta($quiz_id, '_quiz_restrict_to_members', true);
    $restrict_to_products = get_post_meta($quiz_id, '_quiz_restrict_to_products', true);
    $restrict_to_products = is_array($restrict_to_products) ? $restrict_to_products : [];

    // Check restrictions
    $user_id = get_current_user_id();
    $is_restricted_to_members = $restrict_to_members === 'yes';
    $is_restricted_to_products = !empty($restrict_to_products);

    if ($is_restricted_to_members && !$user_id) {
        wp_send_json_error(['message' => __('Please log in to access this quiz.', 'studiare-core')]);
    }

    if ($is_restricted_to_products && !studiare_user_has_purchased_products($user_id, $restrict_to_products)) {
        $product_names = [];
        foreach ($restrict_to_products as $product_id) {
            $product = wc_get_product($product_id);
            if ($product) {
                $product_names[] = $product->get_name();
            }
        }
        wp_send_json_error([
            'message' => sprintf(
                __('You need to purchase one of the following products to access this quiz: %s.', 'studiare-core'),
                implode(', ', $product_names)
            )
        ]);
    }

    // Check if questions exist
    if (empty($questions) || !is_array($questions)) {
        wp_send_json_error(['message' => __('No questions found for this quiz.', 'studiare-core')]);
    }

    // Check quiz timing
    $current_time = current_time('timestamp');
    $start_timestamp = !empty($start_time) ? strtotime($start_time) : 0;
    $end_timestamp = !empty($end_time) ? strtotime($end_time) : PHP_INT_MAX;

    if ($start_timestamp && $current_time < $start_timestamp) {
        wp_send_json_error(['message' => __('Quiz has not started yet.', 'studiare-core')]);
    }

    if ($end_timestamp && $current_time > $end_timestamp) {
        wp_send_json_error(['message' => __('Quiz has ended.', 'studiare-core')]);
    }

    // If pagination is disabled, set questions per page to total questions
    if (!$enable_pagination || $questions_per_page <= 0) {
        $questions_per_page = count($questions);
    }

    // Calculate timer duration in seconds (0 means no timer)
    $timer_seconds = !empty($answer_time) && intval($answer_time) > 0 ? intval($answer_time) * 60 : 0;

    // Check for ongoing quiz attempt
    $remaining_seconds = $timer_seconds;
    if ($timer_seconds > 0 && $user_id) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'studiare_quiz_attempts';
        
        // Check if there's an ongoing attempt
        $ongoing_attempt = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT start_time, timer_seconds FROM $table_name WHERE user_id = %d AND quiz_id = %d AND is_completed = 0 ORDER BY start_time DESC LIMIT 1",
                $user_id,
                $quiz_id
            )
        );

        if ($ongoing_attempt) {
            // Calculate remaining time
            $start_timestamp = strtotime($ongoing_attempt->start_time);
            $elapsed_seconds = $current_time - $start_timestamp;
            $remaining_seconds = max(0, $ongoing_attempt->timer_seconds - $elapsed_seconds);
            
            if ($remaining_seconds <= 0) {
                // Time is up, mark attempt as completed
                $wpdb->update(
                    $table_name,
                    ['is_completed' => 1],
                    ['user_id' => $user_id, 'quiz_id' => $quiz_id, 'start_time' => $ongoing_attempt->start_time],
                    ['%d'],
                    ['%d', '%d', '%s']
                );
                wp_send_json_error(['message' => __('Time is up for this quiz attempt.', 'studiare-core')]);
            }
        } else {
            // Start a new attempt
            $wpdb->insert(
                $table_name,
                [
                    'user_id' => $user_id,
                    'quiz_id' => $quiz_id,
                    'start_time' => current_time('mysql'),
                    'timer_seconds' => $timer_seconds,
                    'is_completed' => 0
                ],
                ['%d', '%d', '%s', '%d', '%d']
            );
        }
    }

    // Generate quiz content
    ob_start();
    ?>
    <?php if ($timer_seconds > 0): ?>
        <div class="quiz-timer" id="quiz-timer-<?php echo esc_attr($quiz_id); ?>" data-timer-seconds="<?php echo esc_attr($remaining_seconds); ?>">
            <i class="fal fa-clock"></i>
            <?php esc_html_e('Time Remaining:', 'studiare-core'); ?> 
            <span class="timer-display"></span>
        </div>
    <?php endif; ?>

    <?php if (!empty($start_time)): ?>
        <p><strong><?php esc_html_e('Start Time:', 'studiare-core'); ?></strong> <?php echo date_i18n("Y/m/d [H:i]",esc_html($start_timestamp)); ?></p>
    <?php endif; ?>

    <?php if (!empty($end_time)): ?>
        <p><strong><?php esc_html_e('End Time:', 'studiare-core'); ?></strong> <?php echo date_i18n("Y/m/d [H:i]",$end_timestamp); ?></p>
    <?php endif; ?>

    <form method="post" id="quiz-form-<?php echo esc_attr($quiz_id); ?>">
        <input type="hidden" name="quiz_id" value="<?php echo esc_attr($quiz_id); ?>">
        <input type="hidden" name="pid" value="<?php echo esc_attr($pid); ?>">
        <div id="quiz-pages">
            <?php
            // Split questions into pages
            $question_chunks = array_chunk($questions, $questions_per_page);
            foreach ($question_chunks as $page_index => $page_questions):
            ?>
                <div class="quiz-page" style="display: <?php echo $page_index === 0 ? 'block' : 'none'; ?>;" data-page="<?php echo esc_attr($page_index); ?>">
                    <?php foreach ($page_questions as $index => $question): ?>
                        <div class="quiz-question">
                            <?php if (!empty($question['image'])): ?>
                                <div class="quiz-question-image">
                                    <img src="<?php echo esc_url($question['image']); ?>" alt="<?php echo esc_attr($question['title']); ?>" style="max-width: 100%; height: auto;">
                                </div>
                            <?php endif; ?>
                            <h4><?php echo esc_html(sprintf(__('Question %d', 'studiare-core'), ($page_index * $questions_per_page) + $index + 1)); ?></h4>
                            <p class="quiz-question-title"><?php echo esc_html($question['title']); ?></p>
                            <?php if (!empty($question['description'])): ?>
                                <div class="quiz-question-description">
                                    <?php echo wp_kses_post($question['description']); ?>
                                </div>
                            <?php endif; ?>
                            <?php foreach (['a', 'b', 'c', 'd'] as $opt): ?>
                                <?php if (!empty($question['option_' . $opt])): ?>
                                    <label>
                                        <input type="radio" name="question_<?php echo esc_attr(($page_index * $questions_per_page) + $index); ?>" value="<?php echo esc_attr($opt); ?>">
                                        <?php echo esc_html($question['option_' . $opt]); ?>
                                    </label>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- Pagination and Submit Controls -->
        <div class="quiz-controls">
            <?php if ($enable_pagination && count($question_chunks) > 1): ?>
                <div class="quiz-pagination">
                    <button class="button" type="button" id="quiz-prev-page-<?php echo esc_attr($quiz_id); ?>" disabled><?php esc_html_e('Previous', 'studiare-core'); ?></button>
                    <button class="button" type="button" id="quiz-next-page-<?php echo esc_attr($quiz_id); ?>"><?php esc_html_e('Next', 'studiare-core'); ?></button>
                    <button class="button submit" type="submit" id="quiz-submit-<?php echo esc_attr($quiz_id); ?>" style="display: none;"><?php esc_html_e('Submit Quiz', 'studiare-core'); ?></button>
                </div>
            <?php else: ?>
                <button class="button submit" type="submit" id="quiz-submit-<?php echo esc_attr($quiz_id); ?>"><?php esc_html_e('Submit Quiz', 'studiare-core'); ?></button>
            <?php endif; ?>
        </div>
    </form>

    <div id="quiz-result-<?php echo esc_attr($quiz_id); ?>"></div>
    <?php
    $html = ob_get_clean();
    wp_send_json_success(['html' => $html]);
}
add_action('wp_ajax_studiare_load_quiz_content', 'studiare_load_quiz_content_ajax');
add_action('wp_ajax_nopriv_studiare_load_quiz_content', 'studiare_load_quiz_content_ajax');

/**
 * Handle quiz submission via AJAX
 */
function studiare_handle_quiz_submission_ajax() {
    // Check if quiz system is enabled
    if (!get_option('studiare_quiz_post_type_enabled', true)) {
        wp_send_json_error(['message' => __('The quiz system is currently disabled.', 'studiare-core')]);
    }

    // Start session if not already started
    if (!session_id() && !headers_sent()) {
        session_start();
    }

    if (!isset($_POST['form_data'])) {
        wp_send_json_error(['message' => __('No form data received.', 'studiare-core')]);
    }

    parse_str($_POST['form_data'], $form_data);
    $quiz_id = isset($form_data['quiz_id']) ? intval($form_data['quiz_id']) : 0;
    $pid = isset($form_data['pid']) ? intval($form_data['pid']) : 0;//javad

    if ($quiz_id <= 0) {
        wp_send_json_error(['message' => __('Invalid quiz ID.', 'studiare-core')]);
    }

    $questions = get_post_meta($quiz_id, '_quiz_questions', true);
    if (empty($questions) || !is_array($questions)) {
        wp_send_json_error(['message' => __('No questions found for this quiz.', 'studiare-core')]);
    }

    // Get score storage method
    $score_storage_method = get_post_meta($quiz_id, '_quiz_score_storage_method', true);
    $score_storage_method = in_array($score_storage_method, ['all_attempts', 'highest_score', 'average_score']) ? $score_storage_method : 'all_attempts';

    $score = 0;
    $total_possible_score = 0;
    $correct_count = 0;
    $incorrect_count = 0;
    $unanswered_count = 0;
    $total_questions = count($questions);

    foreach ($questions as $index => $question) {
        // Add to total possible score
        $question_score = floatval($question['score']);
        $total_possible_score += $question_score;

        $correct_answer = isset($question['correct']) ? $question['correct'] : '';
        $user_answer = isset($form_data["question_$index"]) ? sanitize_text_field($form_data["question_$index"]) : '';

        if ($user_answer === $correct_answer) {
            $score += $question_score;
            $correct_count++;
        } elseif ($user_answer !== '') {
            if (get_post_meta($quiz_id, '_quiz_negative_marking', true) === 'yes') {
                $score -= 0.5;
            }
            $incorrect_count++;
        } else {
            $unanswered_count++;
        }
    }

    // Calculate percentages
    $correct_percentage = $total_questions > 0 ? ($correct_count / $total_questions) * 100 : 0;
    $incorrect_percentage = $total_questions > 0 ? ($incorrect_count / $total_questions) * 100 : 0;
    $unanswered_percentage = $total_questions > 0 ? ($unanswered_count / $total_questions) * 100 : 0;

    // Get passing score
    $passing_score = floatval(get_post_meta($quiz_id, '_quiz_passing_score', true));
    if ($passing_score <= 0) {
        $passing_score = $total_possible_score * 0.5; // Default to 50% of total score if not set
    }

    // Determine pass/fail status
    $status = $score >= $passing_score ? 'passed' : 'failed';

    // Store details for results shortcode
    $details = [
        'total_possible_score' => $total_possible_score,
        'correct_percentage' => $correct_percentage,
        'incorrect_percentage' => $incorrect_percentage,
        'unanswered_percentage' => $unanswered_percentage,
        'status' => $status,
        'passing_score' => $passing_score,
    ];

    // Store the result in session for immediate display
    $_SESSION['quiz_result_' . $quiz_id] = [
        'score' => sprintf(__('%s out of %s', 'studiare-core'), $score, $total_possible_score),
        'correct_percentage' => $correct_percentage,
        'incorrect_percentage' => $incorrect_percentage,
        'unanswered_percentage' => $unanswered_percentage,
        'status' => $status,
    ];

    // Store the result in the database
    $user_id = get_current_user_id();
    if ($user_id) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'studiare_quiz_results';
        $attempt_table = $wpdb->prefix . 'studiare_quiz_attempts';

        // Always store the attempt to count it towards max_attempts
        $wpdb->insert(
            $table_name,
            [
                'user_id' => $user_id,
                'quiz_id' => $quiz_id,
                'product_id' => $pid,//javad
                'score' => $score,
                'timestamp' => current_time('mysql'),
                'details' => maybe_serialize($details),
            ],
            ['%d', '%d', '%d', '%f', '%s', '%s']
        );

        // Mark the attempt as completed
        $wpdb->update(
            $attempt_table,
            ['is_completed' => 1],
            ['user_id' => $user_id, 'quiz_id' => $quiz_id, 'is_completed' => 0],
            ['%d'],
            ['%d', '%d', '%d']
        );

        // Add a flag in details if this score is not the highest (for display purposes)
        if ($score_storage_method === 'highest_score') {
            $previous_score = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT MAX(score) FROM $table_name WHERE user_id = %d AND quiz_id = %d AND id != %d",
                    $user_id,
                    $quiz_id,
                    $wpdb->insert_id // Exclude the newly inserted record
                )
            );

            $details['is_highest_score'] = ($previous_score === null || $score > floatval($previous_score));
            // Update the details of the current record
            $wpdb->update(
                $table_name,
                ['details' => maybe_serialize($details)],
                ['id' => $wpdb->insert_id],
                ['%s'],
                ['%d']
            );
        }
    }

    // Send the result back
    ob_start();
    ?>
    <h3><?php echo esc_html(sprintf(__('Your Score: %s', 'studiare-core'), $_SESSION['quiz_result_' . $quiz_id]['score'])); ?></h3>
    <table class="quiz-stats-table">
        <tr>
            <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($correct_percentage, 1))); ?></td>
            <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($incorrect_percentage, 1))); ?></td>
            <td><?php echo esc_html(sprintf(__('%s%%', 'studiare-core'), number_format($unanswered_percentage, 1))); ?></td>
        </tr>
        <tr>
            <td><?php esc_html_e('Correct', 'studiare-core'); ?></td>
            <td><?php esc_html_e('Incorrect', 'studiare-core'); ?></td>
            <td><?php esc_html_e('Unanswered', 'studiare-core'); ?></td>
        </tr>
    </table>
    <p>
        <?php esc_html_e('Status:', 'studiare-core'); ?> 
        <span class="<?php echo esc_attr($status === 'passed' ? 'status-passed' : 'status-failed'); ?>">
            <?php echo esc_html($status === 'passed' ? __('Passed', 'studiare-core') : __('Failed', 'studiare-core')); ?>
        </span>
    </p>
    <?php if ($score_storage_method === 'highest_score' && $previous_score !== null && $score <= floatval($previous_score)): ?>
        <p><?php esc_html_e('Note: This score is not the highest. The highest score will be displayed in your results.', 'studiare-core'); ?></p>
    <?php endif; ?>
    <?php
    $response = ob_get_clean();
    wp_send_json_success(['html' => $response]);
}
add_action('wp_ajax_studiare_quiz_submit', 'studiare_handle_quiz_submission_ajax');
add_action('wp_ajax_nopriv_studiare_quiz_submit', 'studiare_handle_quiz_submission_ajax');
?>