/*! ============================================================
 * Studiare Read More (from scratch)
 * - UMD module + auto-init via window.STUDIARE_READMORE
 * - Floating button is aligned to the section, not the viewport:
 *   we create a fixed "slot" sized to the section rect and move
 *   the button into it while floating.
 * - Comments in English (as requested).
 * ============================================================ */

(function (root, factory) {
  if (typeof define === 'function' && define.amd) {
    define([], factory);
  } else if (typeof module === 'object' && module.exports) {
    module.exports = factory();
  } else {
    root.StudiareReadMore = factory();
  }
}(typeof self !== 'undefined' ? self : this, function () {

  // ---------- Utilities ----------
  const debounce = (fn, delay = 60) => {
    let t; return (...args) => { clearTimeout(t); t = setTimeout(() => fn(...args), delay); };
  };

  const isElementInViewportVert = (rect, viewportH) =>
    rect.bottom > 0 && rect.top < viewportH;

  // ---------- Core initializer ----------
  function init(targetEl, opts) {
    if (!targetEl) return null;

    const defaults = {
      // texts
      moreText: 'Show More',
      lessText: 'Show Less',

      // collapsed heights (provide one if you don't use per-device)
      collapsedHeight: 300,

      // per-device heights (preferred)
      heights: { mobile: 260, desktop: 340 },
      breakpoint: 768,   // px; keep in sync with CSS

      sticky: true,
      zIndex: 10,

      // internal hooks (set by autoInit)
      __usePerDeviceHeights: false
    };
    const o = Object.assign({}, defaults, opts || {});

    // Resolve collapsed height getter
    const getCollapsedHeight = o.__usePerDeviceHeights
      ? (() => {
          return () => {
            const w = window.innerWidth || document.documentElement.clientWidth;
            return (w < o.breakpoint)
              ? (parseInt(o.heights.mobile, 10)  || 260)
              : (parseInt(o.heights.desktop, 10) || 340);
          };
        })()
      : (() => () => (parseInt(o.collapsedHeight, 10) || 300));

    // DOM structure
    const parent = targetEl.parentElement;
    parent.classList.add('studiare-collapsible-wrapper', 'is-collapsed');
    targetEl.classList.add('studiare-collapsible');

    // Initial collapsed state
    let currentCH = Math.max(0, getCollapsedHeight());
    targetEl.style.maxHeight = currentCH + 'px';
    targetEl.style.overflow = 'hidden';

    // Fade overlay
    const fade = document.createElement('div');
    fade.className = 'studiare-collapsible-fade';
    parent.appendChild(fade);

    // Button
    const btn = document.createElement('div');
    btn.className = 'studiare-readmore-btn';
    btn.setAttribute('role', 'button');
    btn.setAttribute('tabindex', '0');
    btn.setAttribute('aria-expanded', 'false');
    btn.style.zIndex = String(o.zIndex);
    btn.innerHTML = `
      <span class="studiare-readmore-btn__text">${o.moreText}</span>
      <i class="rip rip-l-angle-down" aria-hidden="true"></i>
    `;
    parent.appendChild(btn);

    // Hide button & fade if content is short
    requestAnimationFrame(() => {
      if (targetEl.scrollHeight <= currentCH + 5) {
        btn.style.display = 'none';
        fade.style.display = 'none';
      }
    });

    // Floating slot (created lazily)
    let slot = null;          // fixed container appended to body
    let isFloating = false;   // state flag

    function ensureSlot() {
      if (slot) return slot;
      slot = document.createElement('div');
      slot.className = 'srm-float-slot';
      slot.style.zIndex = String(o.zIndex); // ensure z-index from options
      document.body.appendChild(slot);
      return slot;
    }

    // Move button to slot (fixed, aligned with section)
    function moveBtnToSlot() {
      const s = ensureSlot();
      if (btn.parentElement !== s) {
        s.appendChild(btn);
      }
      isFloating = true;
    }

    // Move button back to its natural parent (absolute)
    function moveBtnToParent() {
      if (btn.parentElement !== parent) {
        parent.appendChild(btn);
      }
      isFloating = false;
    }

    // Core floating logic
    function updateFloating() {
      // If sticky disabled or collapsed: keep absolute in parent
      if (!o.sticky) {
        moveBtnToParent();
        btn.style.position = 'absolute';
        btn.style.left = '50%';
        btn.style.bottom = '15px';
        btn.style.transform = 'translate(-50%, -15px)';
        return;
      }
      const expanded = btn.getAttribute('aria-expanded') === 'true';
      if (!expanded) {
        moveBtnToParent();
        btn.style.position = 'absolute';
        btn.style.left = '50%';
        btn.style.bottom = '15px';
        btn.style.transform = 'translate(-50%, -15px)';
        return;
      }

      const rect = parent.getBoundingClientRect();
      const viewportH = window.innerHeight;

      // If section is completely outside viewport vertically -> no floating
      if (!isElementInViewportVert(rect, viewportH)) {
        moveBtnToParent();
        btn.style.position = 'absolute';
        btn.style.left = '50%';
        btn.style.bottom = '15px';
        btn.style.transform = 'translate(-50%, -15px)';
        return;
      }

      // Natural position logic: if bottom is still below viewport, keep floating
      const btnH = btn.offsetHeight;
      const FLOAT_MARGIN_BOTTOM = 15;
      const naturalBottom = rect.bottom - btnH - FLOAT_MARGIN_BOTTOM;

      if (naturalBottom > viewportH) {
        // FLOAT inside a slot that matches parent's left/width
        moveBtnToSlot();

        const s = ensureSlot();
        // Position slot to mirror the section container
        // getBoundingClientRect() is viewport-relative, works with position: fixed
        s.style.left = Math.round(rect.left) + 'px';
        s.style.width = Math.round(rect.width) + 'px';
        s.style.bottom = FLOAT_MARGIN_BOTTOM + 'px';

        // Button inside slot is static and auto-centered by flexbox
        // (no transforms/left needed)
      } else {
        // Reached natural place -> stick back in parent (absolute)
        moveBtnToParent();
        btn.style.position = 'absolute';
        btn.style.left = '50%';
        btn.style.bottom = '15px';
        btn.style.transform = 'translate(-50%, -15px)';
      }
    }

    // Toggle expand/collapse
    function toggle(force) {
      const expanded = (typeof force === 'boolean')
        ? force
        : (btn.getAttribute('aria-expanded') !== 'true');

      if (expanded) {
        // EXPAND
        targetEl.style.maxHeight = targetEl.scrollHeight + 'px';
        btn.setAttribute('aria-expanded', 'true');
        const span = btn.querySelector('.studiare-readmore-btn__text');
        if (span) span.textContent = o.lessText;
        parent.classList.remove('is-collapsed');

        // Immediately sync floating state
        requestAnimationFrame(updateFloating);
      } else {
        // COLLAPSE
        moveBtnToParent();  // ensure it's back
        isFloating = false;

        currentCH = Math.max(0, getCollapsedHeight());
        targetEl.style.maxHeight = currentCH + 'px';
        btn.setAttribute('aria-expanded', 'false');
        const span = btn.querySelector('.studiare-readmore-btn__text');
        if (span) span.textContent = o.moreText;
        parent.classList.add('is-collapsed');

        // Smooth scroll back to the section top
        parent.scrollIntoView({ behavior: 'smooth', block: 'start' });
      }
    }

    // Events
    btn.addEventListener('click', () => toggle());
    btn.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' || e.key === ' ') {
        e.preventDefault();
        toggle();
      }
    });

    // Scroll/resize -> update floating + collapsed height (if collapsed)
    const onSR = debounce(() => {
      updateFloating();

      if (btn.getAttribute('aria-expanded') !== 'true') {
        const nextCH = Math.max(0, getCollapsedHeight());
        if (nextCH !== currentCH) {
          currentCH = nextCH;
          targetEl.style.maxHeight = currentCH + 'px';
        }
      } else {
        // Recompute expanded height in case content changed and keep floating aligned
        targetEl.style.maxHeight = targetEl.scrollHeight + 'px';
      }
    }, 16);

    window.addEventListener('scroll', onSR, { passive: true });
    window.addEventListener('resize', onSR);

    // Mutation observer to react to content height changes
    const mo = new MutationObserver(() => {
      if (btn.getAttribute('aria-expanded') === 'true') {
        targetEl.style.maxHeight = targetEl.scrollHeight + 'px';
        updateFloating();
      }
    });
    mo.observe(targetEl, { childList: true, subtree: true, characterData: true });

    // Public API
    return {
      toggle,
      destroy() {
        window.removeEventListener('scroll', onSR);
        window.removeEventListener('resize', onSR);
        mo.disconnect();
        if (slot && slot.parentNode) slot.parentNode.removeChild(slot);
        btn.remove();
        fade.remove();
        targetEl.style.maxHeight = '';
        targetEl.style.overflow = '';
        parent.classList.remove('studiare-collapsible-wrapper', 'is-collapsed');
        targetEl.classList.remove('studiare-collapsible');
      }
    };
  }

  // ---------- Auto-init (consumes window.STUDIARE_READMORE) ----------
function autoInit() {
  if (typeof window.STUDIARE_READMORE === 'undefined') return;
  const cfg = window.STUDIARE_READMORE;

  if (cfg.showDesktop === false) document.documentElement.classList.add('studiare-rm-hide-desktop');
  if (cfg.showMobile  === false) document.documentElement.classList.add('studiare-rm-hide-mobile');

  const target = document.querySelector(cfg.targetSelector || '.studi_pro_content_holder');
  if (!target) return;

  const bp = typeof cfg.breakpoint === 'number' ? cfg.breakpoint : 768;
  const showDesktop = cfg.showDesktop !== false;
  const showMobile  = cfg.showMobile  !== false;

  const w = window.innerWidth || document.documentElement.clientWidth;
  const allowedNow = (w >= bp) ? showDesktop : showMobile;
  if (!allowedNow) return;

  const hasPerDevice = cfg.heights && (typeof cfg.heights.mobile !== 'undefined' || typeof cfg.heights.desktop !== 'undefined');

  init(target, {
    moreText: (cfg.texts && cfg.texts.more) || 'Show More',
    lessText: (cfg.texts && cfg.texts.less) || 'Show Less',
    collapsedHeight: cfg.collapsedHeight || 300,
    heights: Object.assign({ mobile: 260, desktop: 340 }, cfg.heights || {}),
    breakpoint: bp,
    __usePerDeviceHeights: !!hasPerDevice,
    sticky: !!cfg.sticky,
    zIndex: cfg.zIndex || 10
  });
}

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', autoInit);
  } else {
    autoInit();
  }

  // Expose manual API for Elementor or custom usage
  return { init };
}));
