<?php
//
// Register custom tab in product edit page
add_filter('woocommerce_product_data_tabs', 'studiare_prerequisite_product_tab');
function studiare_prerequisite_product_tab($tabs) {
    $tabs['studiare_prerequisites'] = array(
        'label'    => __('Prerequisites', 'studiare'),
        'target'   => 'studiare_prerequisite_data',
        'class'    => array(),
        'priority' => 90,
    );
    return $tabs;
}

// Content of the custom tab
add_action('woocommerce_product_data_panels', 'studiare_prerequisite_product_tab_content');
function studiare_prerequisite_product_tab_content() {
    global $post;

    // Load existing values
    $required_ids = get_post_meta($post->ID, '_prerequisite_required_products', true);
    $optional_ids = get_post_meta($post->ID, '_prerequisite_optional_products', true);

    $required_ids = is_array($required_ids) ? $required_ids : array();
    $optional_ids = is_array($optional_ids) ? $optional_ids : array();
    ?>
    <div id="studiare_prerequisite_data" class="panel woocommerce_options_panel">
        <div class="options_group">
            <p class="form-field">
                <label for="prerequisite_required"><?php esc_html_e('Required Products', 'studiare'); ?></label>
                <select id="prerequisite_required" name="prerequisite_required[]" class="wc-enhanced-select" multiple="multiple" style="width:100%;">
                    <?php echo studiare_get_product_options($required_ids); ?>
                </select>
                <span class="description"><?php esc_html_e('Select required products (user must purchase them before buying this product).', 'studiare'); ?></span>
            </p>
            <p class="form-field">
                <label for="prerequisite_optional"><?php esc_html_e('Optional Products', 'studiare'); ?></label>
                <select id="prerequisite_optional" name="prerequisite_optional[]" class="wc-enhanced-select" multiple="multiple" style="width:100%;">
                    <?php echo studiare_get_product_options($optional_ids); ?>
                </select>
                <span class="description"><?php esc_html_e('Select optional (recommended) products.', 'studiare'); ?></span>
            </p>
        </div>
    </div>
    <?php
}

// Render <option> for product select fields
function studiare_get_product_options($selected_ids = array()) {
    $args = array(
        'post_type'      => 'product',
        'posts_per_page' => -1,
        'post_status'    => 'publish',
        'orderby'        => 'title',
        'order'          => 'ASC',
        'fields'         => 'ids',
    );

    $products = get_posts($args);
    $options = '';

    foreach ($products as $product_id) {
        $title = get_the_title($product_id);
        $selected = in_array($product_id, $selected_ids) ? 'selected' : '';
        $options .= '<option value="' . esc_attr($product_id) . '" ' . $selected . '>' . esc_html($title) . '</option>';
    }

    return $options;
}

// Save selected prerequisites
add_action('woocommerce_process_product_meta', 'studiare_save_prerequisite_data');
function studiare_save_prerequisite_data($post_id) {
    $required = isset($_POST['prerequisite_required']) ? array_map('absint', (array) $_POST['prerequisite_required']) : array();
    $optional = isset($_POST['prerequisite_optional']) ? array_map('absint', (array) $_POST['prerequisite_optional']) : array();

    update_post_meta($post_id, '_prerequisite_required_products', $required);
    update_post_meta($post_id, '_prerequisite_optional_products', $optional);
}

// Core function to render prerequisites
function studiare_render_prerequisites($product) {
    // Check if prerequisites system is enabled
    $prerequisites_enabled = class_exists('Redux') ? codebean_option('prerequisites_enabled') : false;
    if (!$prerequisites_enabled) return;

    // Get settings from Redux
    $title_tag = class_exists('Redux') ? codebean_option('prerequisites_title_tag') : 'h2';
    $title_text = class_exists('Redux') ? codebean_option('prerequisites_title_text') : __('Prerequisites', 'studiare');
    $description = class_exists('Redux') ? codebean_option('prerequisites_description') : '';

    $required = get_post_meta($product->get_id(), '_prerequisite_required_products', true);
    $optional = get_post_meta($product->get_id(), '_prerequisite_optional_products', true);

    // Check if there are any prerequisites (required or optional)
    if (empty($required) && empty($optional)) {
        return; // No output if no prerequisites exist
    }

    // Start output buffering
    ob_start();

    // Display title if set
    if (!empty($title_text)) {
        echo '<' . esc_html($title_tag) . ' class="prerequisites-title">' . esc_html($title_text) . '</' . esc_html($title_tag) . '>';
    }

    // Display description if set
    if (!empty($description)) {
        echo '<p class="prerequisites-description">' . wp_kses_post($description) . '</p>';
    }

    // Display prerequisites list
    echo '<ul class="studiare-prerequisites">';
    if (!empty($required)) {
        foreach ($required as $id) {
            $linked = wc_get_product($id);
            if ($linked) {
                echo '<li><a href="' . esc_url(get_permalink($id)) . '">' . esc_html($linked->get_name()) . '</a> (' . __('Required', 'studiare') . ')</li>';
            }
        }
    }
    if (!empty($optional)) {
        foreach ($optional as $id) {
            $linked = wc_get_product($id);
            if ($linked) {
                echo '<li><a href="' . esc_url(get_permalink($id)) . '">' . esc_html($linked->get_name()) . '</a> (' . __('Optional', 'studiare') . ')</li>';
            }
        }
    }
    echo '</ul>';

    // Return buffered content
    return ob_get_clean();
}

// Display prerequisites in product tabs
add_filter('woocommerce_product_tabs', 'studiare_add_prerequisites_tab');
function studiare_add_prerequisites_tab($tabs) {
    global $product;

    // Check if prerequisites system is enabled and set to display in tabs
    $prerequisites_enabled = class_exists('Redux') ? codebean_option('prerequisites_enabled') : false;
    $display_location = class_exists('Redux') ? codebean_option('prerequisites_display_location') : 'in_tabs';

    if (!$prerequisites_enabled || $display_location !== 'in_tabs') {
        return $tabs;
    }

    $required = get_post_meta($product->get_id(), '_prerequisite_required_products', true);
    $optional = get_post_meta($product->get_id(), '_prerequisite_optional_products', true);

    if (!empty($required) || !empty($optional)) {
        $tabs['prerequisites_tab'] = array(
            'title'    => class_exists('Redux') && codebean_option('prerequisites_title_text') ? codebean_option('prerequisites_title_text') : __('Prerequisites', 'studiare'),
            'priority' => 60,
            'callback' => 'studiare_render_prerequisites_tab'
        );
    }

    return $tabs;
}

// Render prerequisites in product tabs
function studiare_render_prerequisites_tab() {
    global $product;
    echo studiare_render_prerequisites($product);
}

// Display prerequisites before add to cart button
add_action('woocommerce_before_add_to_cart_button', 'studiare_display_prerequisites_before_cart');
function studiare_display_prerequisites_before_cart() {
    global $product;

    // Check if prerequisites system is enabled and set to display before add to cart
    $prerequisites_enabled = class_exists('Redux') ? codebean_option('prerequisites_enabled') : false;
    $display_location = class_exists('Redux') ? codebean_option('prerequisites_display_location') : 'in_tabs';

    if ($prerequisites_enabled && $display_location === 'before_add_to_cart') {
        echo studiare_render_prerequisites($product);
    }
}

// Display prerequisites in product content (before or after)
add_filter('the_content', 'studiare_display_prerequisites_in_content');
function studiare_display_prerequisites_in_content($content) {
    global $product;

    // Only apply to single product pages
    if (!is_singular('product') || !is_a($product, 'WC_Product')) {
        return $content;
    }

    // Check if prerequisites system is enabled and set to display in content
    $prerequisites_enabled = class_exists('Redux') ? codebean_option('prerequisites_enabled') : false;
    $display_location = class_exists('Redux') ? codebean_option('prerequisites_display_location') : 'in_tabs';

    if (!$prerequisites_enabled || !in_array($display_location, ['before_content', 'after_content'])) {
        return $content;
    }

    $prerequisites_content = studiare_render_prerequisites($product);

    // Add prerequisites before or after content based on setting
    if ($display_location === 'before_content') {
        return $prerequisites_content . $content;
    } elseif ($display_location === 'after_content') {
        return $content . $prerequisites_content;
    }

    return $content;
}

// Display prerequisites in short description
add_filter('woocommerce_short_description', 'studiare_display_prerequisites_in_short_description');
function studiare_display_prerequisites_in_short_description($description) {
    global $product;

    // Check if prerequisites system is enabled and set to display in short description
    $prerequisites_enabled = class_exists('Redux') ? codebean_option('prerequisites_enabled') : false;
    $display_location = class_exists('Redux') ? codebean_option('prerequisites_display_location') : 'in_tabs';

    if ($prerequisites_enabled && $display_location === 'in_short_description') {
        $prerequisites_content = studiare_render_prerequisites($product);
        return $description . $prerequisites_content;
    }

    return $description;
}

// Prevent add-to-cart if required prerequisites not bought
add_filter('woocommerce_add_to_cart_validation', 'studiare_check_required_prerequisites', 100, 3);
function studiare_check_required_prerequisites($passed, $product_id, $quantity) {
    // Check if prerequisites system is enabled
    $prerequisites_enabled = class_exists('Redux') ? codebean_option('prerequisites_enabled') : false;
    if (!$prerequisites_enabled) return $passed;

    $required = get_post_meta($product_id, '_prerequisite_required_products', true);
    if (empty($required)) return $passed;

    if (!is_user_logged_in()) {
        wc_add_notice(__('You must be logged in to purchase this product.', 'studiare'), 'error');
        return false;
    }

    $user_id = get_current_user_id();
    $missing_titles = [];

    foreach ($required as $req_id) {
        if (!wc_customer_bought_product('', $user_id, $req_id)) {
            $missing_titles[] = get_the_title($req_id);
        }
    }

    if (!empty($missing_titles)) {
        $titles = implode(', ', $missing_titles);
        wc_add_notice(sprintf(__('You must purchase the following prerequisites before buying this product: %s', 'studiare'), $titles), 'error');
        return false;
    }

    return true;
}